/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.common;

import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Files;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.FileOption;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.configuration.Option;
import org.sosy_lab.common.configuration.Options;

@Options(prefix="log", description="Possible log levels in descending order \n(lower levels include higher ones):\nOFF:      no logs published\nSEVERE:   error messages\nWARNING:  warnings\nINFO:     messages\nFINE:     logs on main application level\nFINER:    logs on central CPA algorithm level\nFINEST:   logs published by specific CPAs\nALL:      debugging information\nCare must be taken with levels of FINER or lower, as output files may become quite large and memory usage might become an issue.")
public class LogManager {
    @Option(name="level", toUppercase=true, description="log level of file output")
    private Level fileLevel = Level.OFF;
    @Option(toUppercase=true, description="log level of console output")
    private Level consoleLevel = Level.INFO;
    @Option(toUppercase=true, description="single levels to be excluded from being logged")
    private List<Level> fileExclude = ImmutableList.of();
    @Option(toUppercase=true, description="single levels to be excluded from being logged")
    private List<Level> consoleExclude = ImmutableList.of();
    @Option(name="file", description="name of the log file")
    @FileOption(value=FileOption.Type.OUTPUT_FILE)
    private File outputFile = new File("CPALog.txt");
    @Option(description="maximum size of log output strings before they will be truncated")
    private int truncateSize = 10000;
    private static final Level exceptionDebugLevel = Level.ALL;
    private static final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss:SSS");
    private static final Joiner messageFormat = Joiner.on((char)' ').useForNull("null");
    private final Logger logger;

    private static String extractSimpleClassName(LogRecord lr) {
        String fullClassName = lr.getSourceClassName();
        int dotIndex = fullClassName.lastIndexOf(46);
        assert (dotIndex < fullClassName.length() - 1) : "Last character in a class name cannot be a dot";
        String className = fullClassName.substring(dotIndex + 1);
        return className;
    }

    public LogManager(Configuration config) throws InvalidConfigurationException {
        this(config, new ConsoleHandler());
    }

    public LogManager(Configuration config, Handler consoleOutputHandler) throws InvalidConfigurationException {
        Preconditions.checkNotNull((Object)consoleOutputHandler);
        config.inject(this);
        Level effectiveLogLevel = this.fileLevel.intValue() > this.consoleLevel.intValue() ? this.consoleLevel : this.fileLevel;
        this.logger = Logger.getAnonymousLogger();
        this.logger.setLevel(effectiveLogLevel);
        this.logger.setUseParentHandlers(false);
        if (effectiveLogLevel.equals(Level.OFF)) {
            return;
        }
        this.setupHandler(consoleOutputHandler, new ConsoleLogFormatter(config), this.consoleLevel, this.consoleExclude);
        if (!this.fileLevel.equals(Level.OFF) && this.outputFile != null) {
            try {
                Files.createParentDirs((File)this.outputFile);
                FileHandler outfileHandler = new FileHandler(this.outputFile.getAbsolutePath(), false);
                this.setupHandler(outfileHandler, new FileLogFormatter(), this.fileLevel, this.fileExclude);
            }
            catch (IOException e) {
                if (this.consoleLevel.intValue() > this.fileLevel.intValue()) {
                    this.logger.getHandlers()[0].setLevel(this.fileLevel);
                }
                this.logger.log(Level.WARNING, "Could not open log file " + e.getMessage() + ", redirecting log output to console");
            }
        }
    }

    private void setupHandler(Handler handler, Formatter formatter, Level level, List<Level> excludeLevels) throws InvalidConfigurationException {
        if (excludeLevels.size() > 0) {
            handler.setFilter(new LogLevelFilter(excludeLevels));
        } else {
            handler.setFilter(null);
        }
        handler.setFormatter(formatter);
        handler.setLevel(level);
        this.logger.addHandler(handler);
    }

    public boolean wouldBeLogged(Level priority) {
        return this.logger.isLoggable(priority);
    }

    public void log(Level priority, Object ... args) {
        if (this.wouldBeLogged(priority)) {
            StackTraceElement[] trace = Thread.currentThread().getStackTrace();
            int traceIndex = 2;
            String methodName = trace[traceIndex].getMethodName();
            while (methodName.startsWith("log") || methodName.startsWith("access$")) {
                methodName = trace[++traceIndex].getMethodName();
            }
            this.log0(priority, trace[traceIndex], args);
        }
    }

    private void log0(Level priority, StackTraceElement stackElement, Object ... args) {
        Object[] argsStr = new String[args.length];
        for (int i = 0; i < args.length; ++i) {
            String arg = Objects.firstNonNull((Object)args[i], (Object)"null").toString();
            argsStr[i] = this.truncateSize > 0 && arg.length() > this.truncateSize ? "<ARGUMENT OMITTED BECAUSE " + arg.length() + " CHARACTERS LONG>" : arg;
        }
        LogRecord record = new LogRecord(priority, messageFormat.join(argsStr));
        record.setSourceClassName(stackElement.getClassName());
        record.setSourceMethodName(stackElement.getMethodName());
        this.logger.log(record);
    }

    public void logUserException(Level priority, Throwable e, String additionalMessage) {
        if (this.wouldBeLogged(priority)) {
            String logMessage = "";
            if (priority.equals(Level.SEVERE)) {
                logMessage = "Error: ";
            } else if (priority.equals(Level.WARNING)) {
                logMessage = "Warning: ";
            }
            String exceptionMessage = Strings.nullToEmpty((String)e.getMessage());
            if (Strings.isNullOrEmpty((String)additionalMessage)) {
                logMessage = !exceptionMessage.isEmpty() ? logMessage + exceptionMessage : logMessage + e.getClass().getSimpleName() + " in " + e.getStackTrace()[0];
            } else {
                logMessage = logMessage + additionalMessage;
                if (!exceptionMessage.isEmpty()) {
                    logMessage = e instanceof IOException && logMessage.endsWith("file") ? logMessage + " " + exceptionMessage : logMessage + " (" + exceptionMessage + ")";
                }
            }
            StackTraceElement[] trace = e.getStackTrace();
            int traceIndex = 0;
            if (e instanceof InvalidConfigurationException) {
                while (trace[traceIndex].getClassName().equals(Configuration.class.getName())) {
                    ++traceIndex;
                }
            }
            this.log0(priority, trace[traceIndex], logMessage);
        }
        this.logDebugException(e, additionalMessage);
    }

    public void logDebugException(Throwable e, String additionalMessage) {
        this.logException(exceptionDebugLevel, e, additionalMessage);
    }

    public void logDebugException(Throwable e) {
        this.logDebugException(e, null);
    }

    public void logException(Level priority, Throwable e, String additionalMessage) {
        if (this.wouldBeLogged(priority)) {
            String logMessage = "";
            if (!Strings.isNullOrEmpty((String)additionalMessage)) {
                logMessage = additionalMessage + "\n";
            }
            logMessage = logMessage + Throwables.getStackTraceAsString((Throwable)e);
            StackTraceElement[] trace = Thread.currentThread().getStackTrace();
            int traceIndex = 2;
            while (trace[traceIndex].getMethodName().startsWith("log")) {
                ++traceIndex;
            }
            LogRecord record = new LogRecord(priority, logMessage);
            record.setSourceClassName(trace[traceIndex].getClassName());
            record.setSourceMethodName(trace[traceIndex].getMethodName());
            this.logger.log(record);
        }
    }

    public void flush() {
        for (Handler handler : this.logger.getHandlers()) {
            handler.flush();
        }
    }

    public void close() {
        for (Handler handler : this.logger.getHandlers()) {
            handler.close();
        }
    }

    private static class LogLevelFilter
    implements Filter {
        private final List<Level> excludeLevels;

        public LogLevelFilter(List<Level> excludeLevels) {
            this.excludeLevels = excludeLevels;
        }

        @Override
        public boolean isLoggable(LogRecord pRecord) {
            return !this.excludeLevels.contains(pRecord.getLevel());
        }
    }

    @Options(prefix="log")
    public static class ConsoleLogFormatter
    extends Formatter {
        @Option(description="use colors for log messages on console")
        private boolean useColors = true;

        public ConsoleLogFormatter(Configuration config) throws InvalidConfigurationException {
            config.inject(this);
            if (System.console() == null || System.getProperty("os.name", "").startsWith("Windows")) {
                this.useColors = false;
            }
        }

        @Override
        public String format(LogRecord lr) {
            StringBuffer sb = new StringBuffer();
            if (this.useColors) {
                if (lr.getLevel().equals(Level.WARNING)) {
                    sb.append("\u001b[1m");
                } else if (lr.getLevel().equals(Level.SEVERE)) {
                    sb.append("\u001b[31;1m");
                }
            }
            sb.append(lr.getMessage());
            sb.append(" (");
            sb.append(LogManager.extractSimpleClassName(lr));
            sb.append(".");
            sb.append(lr.getSourceMethodName());
            sb.append(", ");
            sb.append(lr.getLevel().toString());
            sb.append(")");
            if (this.useColors) {
                sb.append("\u001b[m");
            }
            sb.append("\n\n");
            return sb.toString();
        }
    }

    private static class FileLogFormatter
    extends Formatter {
        private FileLogFormatter() {
        }

        @Override
        public String format(LogRecord lr) {
            return dateFormat.format(lr.getMillis()) + "\t " + "level: " + lr.getLevel().toString() + "\t " + LogManager.extractSimpleClassName(lr) + "." + lr.getSourceMethodName() + "\t " + lr.getMessage() + "\n\n";
        }
    }

    public static class StringHandler
    extends Handler {
        private final StringBuilder sb = new StringBuilder();

        @Override
        public void close() {
        }

        @Override
        public void flush() {
        }

        @Override
        public synchronized void publish(LogRecord record) {
            String msg;
            if (!this.isLoggable(record)) {
                return;
            }
            try {
                msg = this.getFormatter().format(record);
            }
            catch (Exception ex) {
                this.reportError(null, ex, 5);
                return;
            }
            try {
                this.sb.append(msg);
            }
            catch (Exception ex) {
                this.reportError(null, ex, 1);
            }
        }

        public String getLog() {
            return this.sb.toString();
        }

        public void clear() {
            this.sb.setLength(0);
            this.sb.trimToSize();
        }
    }
}

