/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.cpachecker.util.predicates.mathsat5;

import com.google.common.collect.ImmutableMap;
import com.google.common.primitives.UnsignedInteger;
import com.google.common.primitives.UnsignedLong;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.sosy_lab.common.Pair;
import org.sosy_lab.cpachecker.core.counterexample.Model;
import org.sosy_lab.cpachecker.exceptions.SolverException;
import org.sosy_lab.cpachecker.util.predicates.interfaces.view.FormulaManagerView;
import org.sosy_lab.cpachecker.util.predicates.mathsat5.Mathsat5FormulaManager;
import org.sosy_lab.cpachecker.util.predicates.mathsat5.Mathsat5NativeApi;

class Mathsat5Model {
    private static final MathContext ROUNDING_PRECISION = MathContext.DECIMAL128;
    private static Pattern BITVECTOR_PATTERN = Pattern.compile("^(\\d+)_(\\d+)$");
    private static Pattern FLOATING_POINT_PATTERN = Pattern.compile("^(\\d+)_(\\d+)_(\\d+)$");

    Mathsat5Model() {
    }

    private static Model.TermType toMathsatType(long e, long mType) {
        if (Mathsat5NativeApi.msat_is_bool_type(e, mType)) {
            return Model.TermType.Boolean;
        }
        if (Mathsat5NativeApi.msat_is_integer_type(e, mType)) {
            return Model.TermType.Integer;
        }
        if (Mathsat5NativeApi.msat_is_rational_type(e, mType)) {
            return Model.TermType.Real;
        }
        if (Mathsat5NativeApi.msat_is_bv_type(e, mType)) {
            return Model.TermType.Bitvector;
        }
        if (Mathsat5NativeApi.msat_is_fp_type(e, mType)) {
            return Model.TermType.FloatingPoint;
        }
        throw new IllegalArgumentException("Given parameter is not a mathsat type!");
    }

    private static Model.Constant toConstant(long env, long variableId) {
        if (!Mathsat5NativeApi.msat_term_is_constant(env, variableId)) {
            throw new IllegalArgumentException("Given mathsat id doesn't correspond to a constant! (" + Mathsat5NativeApi.msat_term_repr(variableId) + ")");
        }
        long declarationId = Mathsat5NativeApi.msat_term_get_decl(variableId);
        String name = Mathsat5NativeApi.msat_decl_get_name(declarationId);
        Model.TermType type = Mathsat5Model.toMathsatType(env, Mathsat5NativeApi.msat_decl_get_return_type(declarationId));
        Pair<String, Integer> nameIndex = FormulaManagerView.parseName(name);
        if (nameIndex.getSecond() != null) {
            return new Model.Variable((String)nameIndex.getFirst(), (Integer)nameIndex.getSecond(), type);
        }
        return new Model.Constant((String)nameIndex.getFirst(), type);
    }

    private static Model.Function toFunction(long env, long pFunctionId) {
        if (Mathsat5NativeApi.msat_term_is_constant(env, pFunctionId)) {
            throw new IllegalArgumentException("Given mathsat id is a variable! (" + Mathsat5NativeApi.msat_term_repr(pFunctionId) + ")");
        }
        long lDeclarationId = Mathsat5NativeApi.msat_term_get_decl(pFunctionId);
        String lName = Mathsat5NativeApi.msat_decl_get_name(lDeclarationId);
        Model.TermType lType = Mathsat5Model.toMathsatType(env, Mathsat5NativeApi.msat_decl_get_return_type(lDeclarationId));
        int lArity = Mathsat5NativeApi.msat_decl_get_arity(lDeclarationId);
        Object[] lArguments = new Object[lArity];
        for (int lArgumentIndex = 0; lArgumentIndex < lArity; ++lArgumentIndex) {
            Object lValue;
            long lArgument = Mathsat5NativeApi.msat_term_get_arg(pFunctionId, lArgumentIndex);
            String lTermRepresentation = Mathsat5NativeApi.msat_term_repr(lArgument);
            long msatType = Mathsat5NativeApi.msat_term_get_type(lArgument);
            if (Mathsat5NativeApi.msat_is_integer_type(env, msatType)) {
                lValue = new BigInteger(lTermRepresentation);
            } else if (Mathsat5NativeApi.msat_is_rational_type(env, msatType)) {
                lValue = Mathsat5Model.parseReal(lTermRepresentation);
            } else if (Mathsat5NativeApi.msat_is_bv_type(env, msatType)) {
                lValue = Mathsat5Model.interpreteBitvector(lTermRepresentation);
            } else if (Mathsat5NativeApi.msat_is_fp_type(env, msatType)) {
                lValue = Mathsat5Model.interpreteFloatingPoint(lTermRepresentation);
            } else {
                throw new NumberFormatException("Unknown number format: " + lTermRepresentation);
            }
            lArguments[lArgumentIndex] = lValue;
        }
        return new Model.Function(lName, lType, lArguments);
    }

    private static Model.AssignableTerm toAssignable(long env, long pTermId) {
        if (!Mathsat5NativeApi.msat_term_is_constant(env, pTermId)) {
            return Mathsat5Model.toFunction(env, pTermId);
        }
        return Mathsat5Model.toConstant(env, pTermId);
    }

    static Model createMathsatModel(long sourceEnvironment, Mathsat5FormulaManager fmgr) throws SolverException {
        Mathsat5NativeApi.ModelIterator lModelIterator;
        ImmutableMap.Builder model = ImmutableMap.builder();
        try {
            lModelIterator = Mathsat5NativeApi.msat_create_ModelIterator(sourceEnvironment);
        }
        catch (IllegalArgumentException e) {
            throw new SolverException("Model iterator could not be created", e);
        }
        while (lModelIterator.hasNext()) {
            Object lValue;
            long[] lModelElement = lModelIterator.next();
            long lKeyTerm = lModelElement[0];
            long lValueTerm = lModelElement[1];
            Model.AssignableTerm lAssignable = Mathsat5Model.toAssignable(sourceEnvironment, lKeyTerm);
            if (!(Mathsat5NativeApi.msat_term_is_number(sourceEnvironment, lValueTerm) || Mathsat5NativeApi.msat_term_is_boolean_constant(sourceEnvironment, lValueTerm) || Mathsat5NativeApi.msat_term_is_false(sourceEnvironment, lValueTerm) || Mathsat5NativeApi.msat_term_is_true(sourceEnvironment, lValueTerm))) {
                throw new IllegalArgumentException("Mathsat term is not a number!");
            }
            String lTermRepresentation = Mathsat5NativeApi.msat_term_repr(lValueTerm);
            switch (lAssignable.getType()) {
                case Boolean: {
                    if (lTermRepresentation.equals("`true`")) {
                        lValue = true;
                        break;
                    }
                    if (lTermRepresentation.equals("`false`")) {
                        lValue = false;
                        break;
                    }
                    throw new IllegalArgumentException("Mathsat unhandled boolean value " + lTermRepresentation);
                }
                case Real: {
                    lValue = Mathsat5Model.parseReal(lTermRepresentation);
                    break;
                }
                case Integer: {
                    lValue = new BigInteger(lTermRepresentation);
                    break;
                }
                case Bitvector: {
                    lValue = Mathsat5Model.interpreteBitvector(lTermRepresentation);
                    break;
                }
                case FloatingPoint: {
                    lValue = Mathsat5Model.interpreteFloatingPoint(lTermRepresentation);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Mathsat term with unhandled type " + (Object)((Object)lAssignable.getType()));
                }
            }
            model.put((Object)lAssignable, lValue);
        }
        lModelIterator.free();
        return new Model((Map<Model.AssignableTerm, Object>)model.build());
    }

    private static Object interpreteBitvector(String lTermRepresentation) {
        Matcher matcher = BITVECTOR_PATTERN.matcher(lTermRepresentation);
        if (!matcher.matches()) {
            throw new NumberFormatException("Unknown bitvector format: " + lTermRepresentation);
        }
        String term = matcher.group(1);
        String lengthValue = matcher.group(2);
        long length = Long.parseLong(lengthValue);
        Number value = length < 64L ? Long.valueOf(term) : new BigInteger(term);
        return value;
    }

    private static Object interpreteFloatingPoint(String lTermRepresentation) {
        Matcher matcher = FLOATING_POINT_PATTERN.matcher(lTermRepresentation);
        if (!matcher.matches()) {
            throw new NumberFormatException("Unknown floating-point format: " + lTermRepresentation);
        }
        int expWidth = Integer.parseInt(matcher.group(2));
        int mantWidth = Integer.parseInt(matcher.group(3));
        if (expWidth == 11 && mantWidth == 52) {
            return Double.longBitsToDouble(UnsignedLong.valueOf((String)matcher.group(1)).longValue());
        }
        if (expWidth == 8 && mantWidth == 23) {
            return Float.valueOf(Float.intBitsToFloat(UnsignedInteger.valueOf((String)matcher.group(1)).intValue()));
        }
        return new BigInteger(matcher.group(1));
    }

    private static Object parseReal(String lTermRepresentation) {
        BigDecimal lValue;
        try {
            lValue = new BigDecimal(lTermRepresentation);
        }
        catch (NumberFormatException e) {
            String[] lNumbers = lTermRepresentation.split("/");
            if (lNumbers.length != 2) {
                throw new NumberFormatException("Unknown number format: " + lTermRepresentation);
            }
            BigDecimal lNumerator = new BigDecimal(lNumbers[0]);
            BigDecimal lDenominator = new BigDecimal(lNumbers[1]);
            lValue = lNumerator.divide(lDenominator, ROUNDING_PRECISION);
        }
        return lValue;
    }
}

