/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.cpachecker.util.resources;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.math.LongMath;
import com.google.common.primitives.Longs;
import java.math.RoundingMode;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import javax.annotation.Nonnull;
import javax.management.JMException;
import org.sosy_lab.common.concurrency.Threads;
import org.sosy_lab.common.configuration.Configuration;
import org.sosy_lab.common.configuration.InvalidConfigurationException;
import org.sosy_lab.common.configuration.Option;
import org.sosy_lab.common.configuration.Options;
import org.sosy_lab.common.configuration.TimeSpanOption;
import org.sosy_lab.common.log.LogManager;
import org.sosy_lab.common.time.TimeSpan;
import org.sosy_lab.cpachecker.core.ShutdownNotifier;
import org.sosy_lab.cpachecker.util.resources.ProcessCpuTimeLimit;
import org.sosy_lab.cpachecker.util.resources.ResourceLimit;
import org.sosy_lab.cpachecker.util.resources.WalltimeLimit;

public final class ResourceLimitChecker {
    private final Thread thread;

    public ResourceLimitChecker(ShutdownNotifier shutdownNotifier, List<ResourceLimit> limits) {
        Preconditions.checkNotNull((Object)shutdownNotifier);
        if (limits.isEmpty() || shutdownNotifier.shouldShutdown()) {
            this.thread = null;
        } else {
            ResourceLimitCheckRunnable runnable = new ResourceLimitCheckRunnable(shutdownNotifier, limits);
            this.thread = Threads.newThread((Runnable)runnable, (String)"Resource limit checker", (boolean)true);
        }
    }

    public void start() {
        if (this.thread != null) {
            this.thread.start();
        }
    }

    public void cancel() {
        if (this.thread != null) {
            this.thread.interrupt();
        }
    }

    public static ResourceLimitChecker fromConfiguration(Configuration config, LogManager logger, ShutdownNotifier notifier) throws InvalidConfigurationException {
        ImmutableList limitsList;
        ResourceLimitOptions options = new ResourceLimitOptions();
        config.inject((Object)options);
        ImmutableList.Builder limits = ImmutableList.builder();
        if (options.walltime.compareTo(TimeSpan.empty()) >= 0) {
            limits.add((Object)WalltimeLimit.fromNowOn(options.walltime));
        }
        if (options.cpuTime.compareTo(TimeSpan.empty()) >= 0) {
            try {
                limits.add((Object)ProcessCpuTimeLimit.fromNowOn(options.cpuTime));
            }
            catch (JMException e) {
                logger.logDebugException((Throwable)e, "Querying cpu time failed");
                logger.log(Level.WARNING, new Object[]{"Your Java VM does not support measuring the cpu time, cpu time threshold disabled."});
            }
        }
        if (!(limitsList = limits.build()).isEmpty()) {
            logger.log(Level.INFO, new Object[]{"Using the following resource limits:", Joiner.on((String)", ").join((Iterable)Lists.transform((List)limitsList, (Function)new Function<ResourceLimit, String>(){

                public String apply(@Nonnull ResourceLimit pInput) {
                    return pInput.getName();
                }
            }))});
        }
        return new ResourceLimitChecker(notifier, (List<ResourceLimit>)limitsList);
    }

    private static class ResourceLimitCheckRunnable
    implements Runnable {
        private static final long PRECISION = TimeUnit.MILLISECONDS.toNanos(500L);
        private final ShutdownNotifier toNotify;
        private final ImmutableList<ResourceLimit> limits;

        ResourceLimitCheckRunnable(ShutdownNotifier pToNotify, List<ResourceLimit> pLimits) {
            this.toNotify = (ShutdownNotifier)Preconditions.checkNotNull((Object)pToNotify);
            this.limits = ImmutableList.copyOf(pLimits);
            Preconditions.checkArgument((!this.limits.isEmpty() ? 1 : 0) != 0);
        }

        @Override
        public void run() {
            ShutdownNotifier.ShutdownRequestListener interruptThreadOnShutdown = ShutdownNotifier.interruptCurrentThreadOnShutdown();
            this.toNotify.registerAndCheckImmediately(interruptThreadOnShutdown);
            long[] timesOfNextCheck = new long[this.limits.size()];
            while (true) {
                long currentTime = System.nanoTime();
                int i = 0;
                for (ResourceLimit limit : this.limits) {
                    if (currentTime < timesOfNextCheck[i]) {
                        ++i;
                        continue;
                    }
                    long currentValue = limit.getCurrentValue();
                    if (limit.isExceeded(currentValue)) {
                        String reason = String.format("The %s has elapsed.", limit.getName());
                        this.toNotify.requestShutdown(reason);
                        return;
                    }
                    long nanosToNextCheck = limit.nanoSecondsToNextCheck(currentValue);
                    timesOfNextCheck[i] = currentTime + nanosToNextCheck;
                    ++i;
                }
                long timeOfNextCheck = Longs.min((long[])timesOfNextCheck);
                long nanosToSleep = Math.max(timeOfNextCheck - currentTime, PRECISION);
                long millisToSleep = LongMath.divide((long)nanosToSleep, (long)1000000L, (RoundingMode)RoundingMode.UP);
                try {
                    Thread.sleep(millisToSleep);
                }
                catch (InterruptedException e) {
                    this.toNotify.unregister(interruptThreadOnShutdown);
                    return;
                }
            }
        }
    }

    @Options(prefix="limits")
    private static class ResourceLimitOptions {
        @Option(secure=true, name="time.wall", description="Limit for wall time used by CPAchecker (use seconds or specify a unit; -1 for infinite)")
        @TimeSpanOption(codeUnit=TimeUnit.NANOSECONDS, defaultUserUnit=TimeUnit.SECONDS, min=-1L)
        private TimeSpan walltime = TimeSpan.ofNanos((long)-1L);
        @Option(secure=true, name="time.cpu", description="Limit for cpu time used by CPAchecker (use seconds or specify a unit; -1 for infinite)")
        @TimeSpanOption(codeUnit=TimeUnit.NANOSECONDS, defaultUserUnit=TimeUnit.SECONDS, min=-1L)
        private TimeSpan cpuTime = TimeSpan.ofNanos((long)-1L);

        private ResourceLimitOptions() {
        }
    }
}

