/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.client.applications.benchmarking;

import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.CheckedFuture;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.client.applications.benchmarking.BenchmarkClientExitCodes;
import org.sosy_lab.verifiercloud.client.applications.benchmarking.BenchmarkingClientException;
import org.sosy_lab.verifiercloud.client.applications.benchmarking.RunCollectionResult;
import org.sosy_lab.verifiercloud.client.network.MasterConnection;
import org.sosy_lab.verifiercloud.client.network.exceptions.FileNotAvailableOnTheMaster;
import org.sosy_lab.verifiercloud.client.network.exceptions.RunCanceledException;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.transportable.file_hierarchy.FileAtRelativePath;
import org.sosy_lab.verifiercloud.transportable.file_hierarchy.RelativePath;
import org.sosy_lab.verifiercloud.transportable.file_hierarchy.string_based.StringRelativePath;
import org.sosy_lab.verifiercloud.transportable.filecontent.FileContent;
import org.sosy_lab.verifiercloud.transportable.info.worker.constant.HostInformation;
import org.sosy_lab.verifiercloud.transportable.run.Run;
import org.sosy_lab.verifiercloud.transportable.run.RunResult;
import org.sosy_lab.verifiercloud.transportable.units.time.TimeInterval;

public class OutputHandler {
    private static final String NEW_FILES_FILENAME = "__CLOUD__created_files.txt";
    private static final String STDERROR_FILEENDING = ".stdError";
    private static final String DATA_FILEENDING = ".data";
    private static final String LOG_FILEENDING = ".log";
    private static final String FILES_DIRECTORY_ENDING = ".files";
    private static final String HOST_INFORMATION_OUTPUT_FILE = "hostInformation.txt";
    private static final TimeInterval FILE_REQUEST_TIMEOUT = TimeInterval.minutes(10L);
    private static final Joiner.MapJoiner RUN_INFORMATION_JOINER = Joiner.on('\n').withKeyValueSeparator("=");
    private static final Joiner COMMAND_JOINER = Joiner.on(' ');
    private final ExecutorService executor;
    private final Logger logger;
    private final boolean printNewFiles;
    private final RelativePath workerOutputFile;
    private final MasterConnection masterConnection;
    private final Path benchmarkResultDirectory;

    public OutputHandler(boolean printNewFiles, MasterConnection masterConnection, ExecutorService executor, Logger logger, Path workerOutputFile, Path benchmarkResultDirectory) {
        this.executor = Preconditions.checkNotNull(executor);
        this.logger = Preconditions.checkNotNull(logger);
        this.printNewFiles = printNewFiles;
        this.masterConnection = Preconditions.checkNotNull(masterConnection);
        this.workerOutputFile = StringRelativePath.from(workerOutputFile);
        Preconditions.checkArgument(!workerOutputFile.isAbsolute());
        this.benchmarkResultDirectory = Preconditions.checkNotNull(benchmarkResultDirectory);
    }

    public void createAndWriteOutput(RunCollectionResult runCollectionResult) throws BenchmarkingClientException {
        final Map<Run, Path> runToOutputFileNameMap = runCollectionResult.getRunToOutputFileNameMap();
        List<CheckedFuture<RunResult, RunCanceledException>> runResultsFutures = runCollectionResult.getRunResults();
        final int numberOfRuns = runResultsFutures.size();
        final CountDownLatch runResults = new CountDownLatch(numberOfRuns);
        final AtomicInteger receivedRunResults = new AtomicInteger(0);
        this.logger.logf(Level.INFO, "Waiting for %d run results.", numberOfRuns);
        final ConcurrentMap<Run, HostInformation> runsWithHost = Maps.newConcurrentMap();
        for (CheckedFuture<RunResult, RunCanceledException> checkedFuture : runResultsFutures) {
            Futures.addCallback(checkedFuture, new FutureCallback<RunResult>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onSuccess(RunResult runResult) {
                    try {
                        Run run = runResult.getRun();
                        Preconditions.checkState(runToOutputFileNameMap.containsKey(run), "%s is not known.", run);
                        runsWithHost.put(run, Preconditions.checkNotNull(runResult.getHostInformation()));
                        Path fileName = (Path)Preconditions.checkNotNull(runToOutputFileNameMap.get(run));
                        Path logFileNameTarget = OutputHandler.this.benchmarkResultDirectory.resolve(fileName);
                        try {
                            OutputHandler.this.writeRunResult(runResult, logFileNameTarget);
                            OutputHandler.this.logger.logf(Level.INFO, "Received run result for run %d of %d.", receivedRunResults.incrementAndGet(), numberOfRuns);
                        }
                        catch (IOException e) {
                            OutputHandler.this.logger.logf(Level.WARNING, "IO problem (%s) while writing run result for run %s: %s", e.getClass().getSimpleName(), run, e.getMessage());
                        }
                    }
                    finally {
                        runResults.countDown();
                    }
                }

                @Override
                public void onFailure(Throwable t) {
                    OutputHandler.this.logger.logf(Level.WARNING, t, "Error getting run result.", new Object[0]);
                    runResults.countDown();
                }
            }, this.executor);
        }
        runResultsFutures.clear();
        try {
            runResults.await();
        }
        catch (InterruptedException e) {
            throw new BenchmarkingClientException("Interrupt while waiting for results.", e, BenchmarkClientExitCodes.ANY_OTHER_FAILURE);
        }
        this.logger.logf(Level.INFO, "Received all results.", new Object[0]);
        this.writeHostInformations(runsWithHost, runToOutputFileNameMap);
    }

    private Path removeLogSuffix(Path path) {
        String pathString = path.toString();
        if (pathString.endsWith(LOG_FILEENDING)) {
            int endIndex = pathString.length() - 4;
            pathString = pathString.substring(0, endIndex);
        }
        return Paths.get(pathString, new String[0]);
    }

    private void writeHostInformations(Map<Run, HostInformation> runsWithHost, Map<Run, Path> runToOutputFileMap) {
        TreeSet<HostInformation> hostInformations = Sets.newTreeSet(runsWithHost.values());
        Path filePath = this.benchmarkResultDirectory.resolve(HOST_INFORMATION_OUTPUT_FILE);
        try (BufferedWriter writer = Files.newBufferedWriter(filePath, Charset.defaultCharset(), StandardOpenOption.CREATE_NEW);){
            for (HostInformation hostInformation : hostInformations) {
                writer.write("name=" + hostInformation.getHostname());
                writer.newLine();
                writer.write("os=" + hostInformation.getOsName() + " " + hostInformation.getOsVersion());
                writer.newLine();
                writer.write("memory=" + hostInformation.getTotalMemory());
                writer.newLine();
                writer.write("cpuName=" + hostInformation.getCPUName());
                writer.newLine();
                writer.write("frequency=" + hostInformation.getCPUFrequency());
                writer.newLine();
                writer.write("cores=" + hostInformation.getNumberOfProcessors());
                writer.newLine();
            }
            writer.newLine();
            writer.newLine();
            for (Map.Entry entry : runsWithHost.entrySet()) {
                String hostname = ((HostInformation)entry.getValue()).getHostname();
                Path outputFile = runToOutputFileMap.get(entry.getKey());
                String runIdentifier = outputFile.getFileName().toString();
                if (runIdentifier.endsWith(LOG_FILEENDING)) {
                    int endIndex = runIdentifier.length() - 4;
                    runIdentifier = runIdentifier.substring(0, endIndex);
                }
                writer.write(String.format("%s\t%s", hostname, runIdentifier));
                writer.newLine();
            }
        }
        catch (IOException e) {
            this.logger.logf(Level.WARNING, e, "File with host information cannot be written: %s", e.getMessage());
        }
    }

    private Optional<FileAtRelativePath> getWorkerOutputFile(RunResult runResult) {
        for (FileAtRelativePath farp : runResult.getResultFiles()) {
            if (!farp.getRelativePath().equals(this.workerOutputFile)) continue;
            return Optional.of(farp);
        }
        return Optional.absent();
    }

    private void writeRunResult(RunResult runResult, Path targetLogPath) throws IOException {
        Path targetPathNoLogSuffix = this.removeLogSuffix(targetLogPath);
        if (this.logger.wouldLog(Level.INFO)) {
            Path niceTargetPath = targetLogPath;
            Path currentWorkingDir = Paths.get("", new String[0]).toAbsolutePath();
            if (niceTargetPath.startsWith(currentWorkingDir)) {
                niceTargetPath = currentWorkingDir.relativize(niceTargetPath);
            }
            this.logger.logf(Level.INFO, "Writing run result to %s", niceTargetPath);
        }
        this.writeRunResultData(runResult, Paths.get(targetLogPath + DATA_FILEENDING, new String[0]));
        this.writeWorkerOutputFile(runResult, targetLogPath);
        this.writeWarnings(runResult, targetLogPath);
        Path filesDir = Paths.get(targetPathNoLogSuffix + FILES_DIRECTORY_ENDING, new String[0]);
        this.writeResultFiles(runResult, filesDir);
        if (this.printNewFiles) {
            this.printNewFiles(runResult, filesDir.resolve(NEW_FILES_FILENAME));
        }
    }

    private void writeRunResultData(RunResult runResult, Path dataPath) throws IOException {
        this.logger.logf(Level.FINE, "Writing the data to %s.", dataPath);
        try (BufferedWriter w = Files.newBufferedWriter(dataPath, Charset.defaultCharset(), StandardOpenOption.CREATE_NEW);){
            if (runResult.wasKilled()) {
                w.append("killed=true");
                w.newLine();
            }
            w.append("outerwalltime=" + runResult.getOuterWallTime().toSecondsAsDouble() + "s");
            w.newLine();
            w.append(RUN_INFORMATION_JOINER.join(runResult.getRunInformation()));
            w.newLine();
            w.append("host=" + runResult.getHostInformation().getHostname());
            w.newLine();
            if (runResult.getEnergy().isPresent()) {
                w.append("energy=" + runResult.getEnergy().get());
                w.newLine();
            }
        }
    }

    private void writeWorkerOutputFile(RunResult runResult, Path logTargetPath) throws IOException {
        Optional<FileAtRelativePath> workerOutputFile = this.getWorkerOutputFile(runResult);
        if (workerOutputFile.isPresent()) {
            CheckedFuture<FileContent, FileNotAvailableOnTheMaster> resultFileFuture = this.masterConnection.getResultFile(workerOutputFile.get().getFileHash());
            try {
                FileContent resultFile = resultFileFuture.checkedGet(FILE_REQUEST_TIMEOUT.toSeconds(), TimeUnit.SECONDS);
                this.logger.logf(Level.FINE, "Writing the output file to %s.", logTargetPath);
                resultFile.writeToPath(logTargetPath);
            }
            catch (FileNotAvailableOnTheMaster e) {
                this.logger.logf(Level.WARNING, e, "Result file %s is not available.", logTargetPath);
            }
            catch (TimeoutException e) {
                this.logger.logf(Level.WARNING, "Timeout while waiting for result file %s: %s", logTargetPath, e.getMessage());
            }
        } else {
            this.logger.logf(Level.WARNING, "Result file %s not found.", logTargetPath);
        }
    }

    private void writeResultFiles(RunResult runResult, Path filesTargetPath) throws IOException {
        for (FileAtRelativePath resultFile : runResult.getResultFiles()) {
            if (resultFile.getRelativePath().equals(this.workerOutputFile)) continue;
            CheckedFuture<FileContent, FileNotAvailableOnTheMaster> resultFileFuture = this.masterConnection.getResultFile(resultFile.getFileHash());
            Path targetFile = filesTargetPath.resolve(resultFile.getRelativePathString());
            if (targetFile.getParent() != null) {
                Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
            }
            try {
                FileContent resultFileContent = resultFileFuture.checkedGet(FILE_REQUEST_TIMEOUT.toSeconds(), TimeUnit.SECONDS);
                this.logger.logf(Level.FINE, "Writing result file to %s.", targetFile);
                resultFileContent.writeToPath(targetFile);
            }
            catch (FileNotAvailableOnTheMaster e) {
                this.logger.logf(Level.WARNING, e, "Result file %s is not available", resultFile);
            }
            catch (IOException e) {
                this.logger.logf(Level.WARNING, "IO problem while writing result file %s: %s", targetFile, e.getMessage());
            }
            catch (TimeoutException e) {
                this.logger.logf(Level.WARNING, "Timeout while waiting for result file %s: %s", targetFile, e.getMessage());
            }
        }
    }

    private void writeWarnings(RunResult runResult, Path targetPath) throws IOException {
        if (!runResult.getWarnings().getFileSize().isZero()) {
            Path errorPath = Paths.get(targetPath + STDERROR_FILEENDING, new String[0]);
            this.logger.logf(Level.FINE, "Writing warnings to %s.", errorPath);
            runResult.getWarnings().writeToPath(errorPath);
        }
    }

    private void printNewFiles(RunResult runResult, Path targetFile) throws IOException {
        if (targetFile.getParent() != null) {
            Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
        }
        try (BufferedWriter writer = Files.newBufferedWriter(targetFile, Charset.defaultCharset(), StandardOpenOption.CREATE_NEW);){
            this.logger.logf(Level.FINE, "Writing list of all produced output files to %s.", targetFile);
            String explanationString = String.format("# The following files (with SHA-1 hashes in braces)%n# were created during the execution of the run%n# %s%n# on the worker.%n", COMMAND_JOINER.join(runResult.getRun().getCommand()));
            writer.append(explanationString);
            for (FileAtRelativePath farp : runResult.getAllNewFiles()) {
                writer.append(String.format("%s\t(%s)", farp.getRelativePathString(), farp.getFileHash()));
                writer.newLine();
            }
        }
    }
}

