/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.client.applications.cpachecker;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.CheckedFuture;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.client.applications.Client;
import org.sosy_lab.verifiercloud.client.applications.cpachecker.CPAcheckerRunCollectionProvider;
import org.sosy_lab.verifiercloud.client.files.ClientFileStorage;
import org.sosy_lab.verifiercloud.client.network.ClientAuthorizationState;
import org.sosy_lab.verifiercloud.client.network.MasterConnection;
import org.sosy_lab.verifiercloud.client.network.Success;
import org.sosy_lab.verifiercloud.client.network.exceptions.AuthenticationFailedException;
import org.sosy_lab.verifiercloud.client.network.exceptions.FileNotAvailableOnTheMaster;
import org.sosy_lab.verifiercloud.client.network.exceptions.InsufficientRightsException;
import org.sosy_lab.verifiercloud.client.network.exceptions.RunCanceledException;
import org.sosy_lab.verifiercloud.client.network.exceptions.RunSubmissionFailureException;
import org.sosy_lab.verifiercloud.client.run_builders.RunCollectionCreationException;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.permanent_storage.exceptions.PermanentStorageException;
import org.sosy_lab.verifiercloud.global.util.FileUtils;
import org.sosy_lab.verifiercloud.global.util.system.SystemInformationProvider;
import org.sosy_lab.verifiercloud.master.clientside.ClientAccessLevel;
import org.sosy_lab.verifiercloud.transportable.collections.RunCollection;
import org.sosy_lab.verifiercloud.transportable.file_hierarchy.FileAtRelativePath;
import org.sosy_lab.verifiercloud.transportable.filecontent.FileContent;
import org.sosy_lab.verifiercloud.transportable.run.Run;
import org.sosy_lab.verifiercloud.transportable.run.RunResult;
import org.sosy_lab.verifiercloud.transportable.units.time.TimeInterval;

public class CPAcheckerClient
extends Client {
    private static final TimeInterval FILE_REQUEST_TIMEOUT = TimeInterval.minutes(1L);
    private final Thread.UncaughtExceptionHandler uncaughtExceptionHandler;
    private final SystemInformationProvider sysInfoProvider;
    private final ClientFileStorage clientFileStorage;
    private final Path outputDir;
    private final boolean detachable;
    private CPAcheckerRunCollectionProvider runCollectionProvider;
    private Path inputFile;

    @Inject
    public CPAcheckerClient(MasterConnection masterConnection, Logger logger, ClientFileStorage clientFileStorage, SystemInformationProvider sysInfoProvider, Thread.UncaughtExceptionHandler uncaughtExceptionHandler, CPAcheckerRunCollectionProvider runCollectionProvider, @Named(value="output-dir") Optional<Path> outputDir, @Named(value="cpachecker-dir") Path cpacheckerDir, @Named(value="input") Path inputFile, @Named(value="detach") Boolean detachable) {
        super(masterConnection, logger);
        this.clientFileStorage = Preconditions.checkNotNull(clientFileStorage);
        this.sysInfoProvider = Preconditions.checkNotNull(sysInfoProvider);
        this.uncaughtExceptionHandler = Preconditions.checkNotNull(uncaughtExceptionHandler);
        this.runCollectionProvider = Preconditions.checkNotNull(runCollectionProvider);
        this.outputDir = outputDir.isPresent() ? outputDir.get() : cpacheckerDir;
        this.inputFile = inputFile;
        this.detachable = detachable;
    }

    @Override
    public void start() {
        super.start();
        if (!Files.isRegularFile(this.inputFile, new LinkOption[0])) {
            this.logger.logf(Level.SEVERE, "Input file is not an actual file: %s", this.inputFile);
            this.stop();
            return;
        }
        if (!this.masterConnection.isConnected()) {
            return;
        }
        Thread.currentThread().setUncaughtExceptionHandler(this.uncaughtExceptionHandler);
        try {
            this.clientFileStorage.initialize();
        }
        catch (IOException | PermanentStorageException e) {
            this.logger.logf(Level.SEVERE, "Failed to initialize file storage: %s", e.getMessage());
            this.stop();
            return;
        }
        try {
            this.authenticate();
        }
        catch (AuthenticationFailedException e) {
            this.logger.logf(Level.SEVERE, "Failed to authenticate: %s", e.getMessage());
            this.stop();
            return;
        }
        try {
            this.executeCPAchecker();
        }
        catch (CPAcheckerExecutionException e) {
            this.logger.logf(Level.SEVERE, e.getMessage(), new Object[0]);
        }
        this.stop();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void executeCPAchecker() throws CPAcheckerExecutionException {
        CheckedFuture<RunResult, RunCanceledException> futureResult;
        RunCollection runCollection;
        try {
            runCollection = this.runCollectionProvider.createCPAcheckerRunCollection();
        }
        catch (IOException | RunCollectionCreationException e) {
            throw new CPAcheckerExecutionException(e);
        }
        ImmutableList<Run> runs = runCollection.getRuns();
        Preconditions.checkState(runs.size() == 1, "Expected exactly one run, but there are %s", runs.size());
        Run run = (Run)runs.get(0);
        if (this.detachable) {
            CheckedFuture<Success, RunSubmissionFailureException> submitRunCollection = this.masterConnection.submitRunCollection(runCollection);
            try {
                submitRunCollection.checkedGet();
                this.logger.logf(Level.INFO, "Run with ID %s successfully scheduled.", run.getUniqueSecureId());
                return;
            }
            catch (RunSubmissionFailureException e) {
                throw new CPAcheckerExecutionException(e);
            }
        }
        try {
            futureResult = this.sendRun(runCollection, run);
        }
        catch (InsufficientRightsException e) {
            throw new CPAcheckerExecutionException(e);
        }
        RunResult runResult = this.getRunResult(futureResult);
        if (runResult.getExitCode() != 0) {
            this.logger.logf(Level.WARNING, "Exit code of CPAchecker run: %d", runResult.getExitCode());
        }
        String inputFilename = this.inputFile.getFileName().toString();
        if (Files.exists(this.outputDir, new LinkOption[0])) {
            if (!Files.isDirectory(this.outputDir, new LinkOption[0])) throw new CPAcheckerExecutionException("Output directory " + this.outputDir + " is not a directory.");
            if (!Files.isWritable(this.outputDir)) {
                throw new CPAcheckerExecutionException("Output directory " + this.outputDir + " is not writable.");
            }
        } else {
            try {
                Files.createDirectories(this.outputDir, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new CPAcheckerExecutionException(e);
            }
        }
        Optional<Object> outputContent = Optional.absent();
        for (FileAtRelativePath resultFile : runResult.getResultFiles()) {
            if (!resultFile.getRelativePath().equals(CPAcheckerRunCollectionProvider.OUTPUT_FILE)) continue;
            CheckedFuture<FileContent, FileNotAvailableOnTheMaster> outputFuture = this.masterConnection.getResultFile(resultFile.getFileHash());
            try {
                outputContent = Optional.of(outputFuture.checkedGet());
            }
            catch (FileNotAvailableOnTheMaster e) {
                throw new CPAcheckerExecutionException(e);
            }
        }
        if (outputContent.isPresent() && !((FileContent)outputContent.get()).getContent().isEmpty()) {
            Path outputPath = this.outputDir.resolve(inputFilename + ".log");
            try {
                ((FileContent)outputContent.get()).writeToPath(outputPath);
                this.logger.logf(Level.INFO, "Wrote stdout to %s", outputPath);
            }
            catch (IOException e) {
                this.logger.logf(Level.WARNING, "Failed to write output: %s", e.getMessage());
            }
        } else {
            this.logger.logf(Level.WARNING, "CPAchecker run produced no output.", new Object[0]);
        }
        FileContent stderr = runResult.getWarnings();
        if (!stderr.getFileSize().isZero()) {
            Path stderrPath = this.outputDir.resolve(inputFilename + ".stderr");
            try {
                stderr.writeToPath(stderrPath);
                this.logger.logf(Level.INFO, "Wrote stderr to %s", stderrPath);
            }
            catch (IOException e) {
                this.logger.logf(Level.WARNING, "Failed to write stderr: %s", e.getMessage());
            }
        }
        this.getAndWriteOutputFiles(this.outputDir, runResult);
    }

    private void getAndWriteOutputFiles(Path targetDir, RunResult runResult) {
        for (FileAtRelativePath resultFile : runResult.getResultFiles()) {
            FileContent fileContent;
            CheckedFuture<FileContent, FileNotAvailableOnTheMaster> futureFile = this.masterConnection.getResultFile(resultFile.getFileHash());
            try {
                fileContent = futureFile.checkedGet(FILE_REQUEST_TIMEOUT.toSeconds(), TimeUnit.SECONDS);
            }
            catch (TimeoutException | FileNotAvailableOnTheMaster e) {
                this.logger.logf(Level.SEVERE, e, "Unable to get file: %s", e.getMessage());
                continue;
            }
            Path resultPath = targetDir.resolve(resultFile.getRelativePathString());
            try {
                Files.createDirectories(resultPath.getParent(), new FileAttribute[0]);
                FileUtils.deleteFileIfItExists(resultPath);
                fileContent.writeToPath(resultPath);
                this.logger.logf(Level.FINE, "Wrote file %s to %s", resultFile.getRelativePathString(), resultPath);
            }
            catch (IOException e) {
                this.logger.logf(Level.WARNING, e, "Unable to write file: %s.", e.getMessage());
            }
        }
    }

    private RunResult getRunResult(CheckedFuture<RunResult, RunCanceledException> futureRun) throws CPAcheckerExecutionException {
        try {
            return futureRun.checkedGet();
        }
        catch (RunCanceledException e) {
            throw new CPAcheckerExecutionException(e);
        }
    }

    private CheckedFuture<RunResult, RunCanceledException> sendRun(RunCollection runCollection, Run run) throws InsufficientRightsException {
        List<CheckedFuture<RunResult, RunCanceledException>> futureRunResults = this.masterConnection.sendRunCollection(runCollection, this.detachable);
        Preconditions.checkState(futureRunResults.size() == 1, "Expected only one run, but there are %s", futureRunResults.size());
        return futureRunResults.get(0);
    }

    @Override
    public void onConnectionLost() {
        this.logger.logf(Level.SEVERE, "Connection to master closed.", new Object[0]);
        this.stop();
    }

    private void authenticate() throws AuthenticationFailedException {
        CheckedFuture<ClientAuthorizationState, AuthenticationFailedException> authState = this.masterConnection.authenticate(ClientAccessLevel.USER, this.sysInfoProvider.getUsername());
        try {
            switch (authState.checkedGet(5L, TimeUnit.SECONDS)) {
                case DISCONNECTED: {
                    throw new AuthenticationFailedException("Connection lost.");
                }
                case UNAUTHORIZED: {
                    throw new AuthenticationFailedException("Authorization failed!");
                }
            }
        }
        catch (TimeoutException e) {
            throw new AuthenticationFailedException("Timeout when waiting for response from Master.");
        }
    }

    private static final class CPAcheckerExecutionException
    extends Exception {
        private static final long serialVersionUID = 9036479151289273867L;

        public CPAcheckerExecutionException(Exception e) {
            super(e);
        }

        public CPAcheckerExecutionException(String message) {
            super(message);
        }
    }
}

