/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.global.file_storage;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.MapMaker;
import com.google.common.hash.HashCode;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.global.file_storage.FileNotAvailableException;
import org.sosy_lab.verifiercloud.global.file_storage.FileStorage;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.util.FileUtils;
import org.sosy_lab.verifiercloud.transportable.info.FileStorageInformation;
import org.sosy_lab.verifiercloud.transportable.units.memory.MemoryUnit;

public abstract class GenericFileStorage<T>
implements FileStorage<T> {
    public static final String FILE_STORAGE_FOLDER_NAME = "files";
    private static final int INDEXING_LOG_THRESHOLD = 10000;
    private Optional<Path> fileDirectory = Optional.absent();
    private final Map<HashCode, Path> indexOfStoredFiles = new MapMaker().initialCapacity(10000).makeMap();
    private final Map<HashCode, MemoryUnit> fileSizes = new MapMaker().initialCapacity(10000).makeMap();
    protected final Logger logger;

    public GenericFileStorage(Logger logger) {
        this.logger = Preconditions.checkNotNull(logger);
    }

    @Override
    public void initializeFileStorage(Path baseWorkingDir) throws IOException {
        Preconditions.checkState(!this.fileDirectory.isPresent(), "%s must not be initialized more than once.", this.getClass().getSimpleName());
        Preconditions.checkNotNull(baseWorkingDir);
        Preconditions.checkArgument(Files.isDirectory(baseWorkingDir, new LinkOption[0]), "%s is not a directory.", baseWorkingDir);
        this.fileDirectory = Optional.of(baseWorkingDir.resolve(FILE_STORAGE_FOLDER_NAME));
        Files.createDirectories(this.fileDirectory.get(), new FileAttribute[0]);
        this.logger.logf(Level.INFO, "Initialized file storage at %s ", this.fileDirectory.get());
    }

    public void indexFiles() throws IOException {
        this.checkIsInitialized();
        this.indexFiles(this.fileDirectory.get());
        this.logger.logf(Level.INFO, "Indexed file storage at %s with %d files", this.fileDirectory.get(), this.indexOfStoredFiles.size());
    }

    private void indexFiles(Path dir) throws IOException {
        Preconditions.checkArgument(Files.isDirectory(dir, new LinkOption[0]));
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir, new IgnoreHiddenPathsFilter());){
            for (Path child : stream) {
                int currentSize;
                if (Files.isRegularFile(child, new LinkOption[0])) {
                    this.indexFile(child);
                } else if (Files.isDirectory(child, new LinkOption[0])) {
                    this.indexFiles(child);
                } else {
                    this.logger.logf(Level.WARNING, "%s: Ignoring %s.", this.getClass(), child);
                }
                if ((currentSize = this.getNumberOfStoredFile()) <= 0 || currentSize % 10000 != 0) continue;
                this.logger.logf(Level.FINER, "Indexed %s files in file storage.", currentSize);
            }
        }
    }

    protected abstract void indexFile(Path var1) throws IOException;

    protected void addFile(HashCode fileHash, Path file, MemoryUnit fileSize) {
        this.checkIsInitialized();
        Preconditions.checkArgument(!this.fileSizes.containsKey(fileHash));
        Preconditions.checkArgument(!this.indexOfStoredFiles.containsKey(fileHash));
        Preconditions.checkArgument(file.startsWith(this.getFileDirectory()));
        Path relativePath = this.getFileDirectory().relativize(file);
        this.fileSizes.put(fileHash, fileSize);
        this.indexOfStoredFiles.put(fileHash, relativePath);
    }

    protected Path getPathFromStorage(HashCode fileHash) {
        this.checkIsInitialized();
        Preconditions.checkNotNull(fileHash);
        Preconditions.checkArgument(this.indexOfStoredFiles.containsKey(fileHash), "File for hash %s is not stored in %s", fileHash, this.getClass());
        Path relativePath = this.indexOfStoredFiles.get(fileHash);
        return this.getFileDirectory().resolve(relativePath);
    }

    protected int getNumberOfStoredFile() {
        this.checkIsInitialized();
        return this.indexOfStoredFiles.size();
    }

    public Path getFileDirectory() {
        this.checkIsInitialized();
        return this.fileDirectory.get();
    }

    @Override
    public boolean isFileKnown(HashCode fileHash) {
        this.checkIsInitialized();
        Preconditions.checkNotNull(fileHash);
        return this.indexOfStoredFiles.containsKey(fileHash);
    }

    public void removeFile(HashCode fileHash) throws FileNotAvailableException {
        this.checkIsInitialized();
        Preconditions.checkNotNull(fileHash);
        Path relativePath = this.indexOfStoredFiles.remove(fileHash);
        this.fileSizes.remove(fileHash);
        if (relativePath != null) {
            Path file = this.getFileDirectory().resolve(relativePath);
            try {
                FileUtils.deleteFileIfItExists(file);
            }
            catch (IOException e) {
                this.logger.logf(Level.WARNING, e, "Cannot remove file content %s: %s", file, e.getMessage());
            }
        } else {
            throw new FileNotAvailableException(fileHash);
        }
    }

    public void checkIsInitialized() {
        Preconditions.checkState(this.fileDirectory.isPresent(), "%s must be initialized before calling this method.", this.getClass());
    }

    @Override
    public FileStorageInformation getFileStorageInformation() {
        this.checkIsInitialized();
        return new FileStorageInformation(this.indexOfStoredFiles.size(), this.getSizeOfStoredFiles(), this.getAvailableDiskSpace());
    }

    private MemoryUnit getAvailableDiskSpace() {
        MemoryUnit freeSpace;
        Preconditions.checkState(this.fileDirectory.isPresent());
        Path dir = this.fileDirectory.get();
        try {
            FileStore partiton = Files.getFileStore(dir);
            freeSpace = MemoryUnit.bytes(partiton.getUnallocatedSpace());
        }
        catch (IOException e) {
            this.logger.logf(Level.SEVERE, e, "IOException when getting FileStore: %s", e.getMessage());
            throw new RuntimeException(e);
        }
        return freeSpace;
    }

    private MemoryUnit getSizeOfStoredFiles() {
        return MemoryUnit.sum(this.fileSizes.values());
    }

    private static class IgnoreHiddenPathsFilter
    implements DirectoryStream.Filter<Path> {
        private IgnoreHiddenPathsFilter() {
        }

        @Override
        public boolean accept(Path entry) throws IOException {
            return !Files.isHidden(entry);
        }
    }
}

