/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.master.clientside.results;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.hash.HashCode;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.util.FileUtils;
import org.sosy_lab.verifiercloud.master.clientside.results.RunResultStorage;
import org.sosy_lab.verifiercloud.master.clientside.results.RunResultStorageInitializationException;
import org.sosy_lab.verifiercloud.master.files.ConcurrentMasterFileStorage;
import org.sosy_lab.verifiercloud.transportable.file_hierarchy.FileAtRelativePath;
import org.sosy_lab.verifiercloud.transportable.filecontent.FileContent;
import org.sosy_lab.verifiercloud.transportable.info.worker.constant.HostInformation;
import org.sosy_lab.verifiercloud.transportable.run.RunResult;
import org.sosy_lab.verifiercloud.transportable.run.constraints.limitations.Limitations;
import org.sosy_lab.verifiercloud.transportable.units.memory.MemoryUnit;

@Singleton
public class ZipRunResultStorage
implements RunResultStorage {
    private static final String RUN_DESCRIPTION_FILE = "runDescription.txt";
    private static final String RUN_INFORMATION_OUTPUT_FILE = "runInformation.txt";
    private static final String HOST_INFORMATION_OUTPUT_FILE = "hostInformation.txt";
    private static final Joiner.MapJoiner RUN_INFORMATION_JOINER = Joiner.on('\n').withKeyValueSeparator("=");
    private static final Joiner COMMAND_JOINER = Joiner.on(" ");
    private final ConcurrentMasterFileStorage fileStorage;
    private final Logger logger;
    private final ExecutorService executor;
    private final Path folderPath;

    @Inject
    public ZipRunResultStorage(ConcurrentMasterFileStorage fileStorage, Thread.UncaughtExceptionHandler uncaughtExceptionHandler, @Named(value="zip-storage-location") Optional<Path> zipFolder, Logger logger) {
        this.logger = Preconditions.checkNotNull(logger);
        this.fileStorage = Preconditions.checkNotNull(fileStorage);
        Preconditions.checkArgument(zipFolder.isPresent());
        this.folderPath = zipFolder.get();
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setUncaughtExceptionHandler(uncaughtExceptionHandler).setNameFormat(this.getClass().getSimpleName().toString() + "-%d").build();
        this.executor = Executors.newSingleThreadExecutor(threadFactory);
    }

    @Override
    public void initialize() throws RunResultStorageInitializationException {
        if (Files.exists(this.folderPath, new LinkOption[0])) {
            if (!Files.isDirectory(this.folderPath, new LinkOption[0])) {
                throw new RunResultStorageInitializationException("Directory for zip files of run results exists, but is no directory: " + this.folderPath);
            }
        } else {
            try {
                Files.createDirectories(this.folderPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new RunResultStorageInitializationException(e);
            }
        }
        if (!Files.isWritable(this.folderPath)) {
            throw new RunResultStorageInitializationException("Directory for zip files of run results is not writable: " + this.folderPath);
        }
        this.logger.logf(Level.INFO, "Initialized zip file storage at %s.", this.folderPath);
    }

    @Override
    public void storeResult(final RunResult runResult) {
        Preconditions.checkNotNull(runResult);
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                ZipRunResultStorage.this.writeToZipFile(runResult);
            }
        });
    }

    private void writeToZipFile(RunResult runResult) {
        String runId = runResult.getRun().getUniqueSecureId();
        Path zipFilePath = this.getPathFromId(runId);
        URI uri = URI.create("jar:file:" + zipFilePath.toUri().getPath());
        HashMap<String, String> env = Maps.newHashMap();
        env.put("create", "true");
        try (FileSystem zipFile = FileSystems.newFileSystem(uri, env);){
            Optional<Integer> coreLimit;
            Optional<String> possibleDescription;
            int filecount = 0;
            FileContent stderr = runResult.getWarnings();
            if (!stderr.getFileSize().isZero()) {
                Path stderrPath = zipFile.getPath("stderr", new String[0]);
                stderr.writeToPath(stderrPath);
                ++filecount;
            }
            if ((possibleDescription = runResult.getRun().getOptionalDescription()).isPresent()) {
                String description = possibleDescription.get();
                Path descriptionPath = zipFile.getPath(RUN_DESCRIPTION_FILE, new String[0]);
                Files.write(descriptionPath, description.getBytes(Charsets.UTF_8), new OpenOption[0]);
            }
            StringBuilder runInfo = new StringBuilder();
            runInfo.append("command=");
            runInfo.append(COMMAND_JOINER.join(runResult.getRun().getCommand())).append('\n');
            if (runResult.getEnergy().isPresent()) {
                runInfo.append("energy=");
                runInfo.append(runResult.getEnergy().get().toJoule()).append("J\n");
            }
            runInfo.append(RUN_INFORMATION_JOINER.join(runResult.getRunInformation()));
            runInfo.append("\n");
            Limitations limitations = runResult.getRun().getLimitations();
            runInfo.append("timeLimit=");
            runInfo.append(limitations.getTimeLimit().toSecondsAsDouble()).append("s\n");
            Optional<MemoryUnit> memoryLimit = limitations.getMemoryLimit();
            if (memoryLimit.isPresent()) {
                runInfo.append("memoryLimit=");
                runInfo.append(memoryLimit.get().toByte()).append("B\n");
            }
            if ((coreLimit = limitations.getProcessorLimit()).isPresent()) {
                runInfo.append("coreLimit=");
                runInfo.append(coreLimit.get()).append('\n');
            }
            Path runInfoPath = zipFile.getPath(RUN_INFORMATION_OUTPUT_FILE, new String[0]);
            Files.write(runInfoPath, runInfo.toString().getBytes(Charsets.UTF_8), new OpenOption[0]);
            StringBuilder hostInfo = new StringBuilder();
            hostInfo.append("name=");
            HostInformation hostInformation = runResult.getHostInformation();
            hostInfo.append(hostInformation.getHostname()).append('\n');
            hostInfo.append("os=");
            hostInfo.append(hostInformation.getOsName()).append(' ');
            hostInfo.append(hostInformation.getOsVersion()).append('\n');
            hostInfo.append("memory=");
            hostInfo.append(hostInformation.getTotalMemory().toByte()).append('\n');
            hostInfo.append("cpuModel=");
            hostInfo.append(hostInformation.getCPUName()).append('\n');
            hostInfo.append("frequency=");
            hostInfo.append(hostInformation.getCPUFrequency()).append('\n');
            hostInfo.append("cores=");
            hostInfo.append(hostInformation.getNumberOfProcessors()).append('\n');
            Path hostInfoPath = zipFile.getPath(HOST_INFORMATION_OUTPUT_FILE, new String[0]);
            Files.write(hostInfoPath, hostInfo.toString().getBytes(Charsets.UTF_8), new OpenOption[0]);
            for (FileAtRelativePath farp : runResult.getResultFiles()) {
                Path filePath = zipFile.getPath(farp.getRelativePathString(), new String[0]);
                Path parent = filePath.getParent();
                if (parent != null) {
                    Files.createDirectories(parent, new FileAttribute[0]);
                }
                HashCode fileHash = farp.getFileHash();
                ListenableFuture<FileContent> fileContent = this.fileStorage.getFileContent(fileHash);
                FileContent file = (FileContent)fileContent.get();
                file.writeToPath(filePath);
                ++filecount;
            }
            this.logger.logf(Level.FINEST, "Logged result zip for %s to %s with %d files.", runResult.getRun(), zipFilePath, filecount);
        }
        catch (IOException | InterruptedException | ExecutionException e) {
            this.logger.logf(Level.WARNING, e, "Failed to write result zip file for %s to %s: %s", runResult.getRun(), zipFilePath, e.getMessage());
        }
    }

    @Override
    public boolean isResultAvailable(final String runId) {
        Future<Boolean> future = this.executor.submit(new Callable<Boolean>(){

            @Override
            public Boolean call() throws Exception {
                return Files.isRegularFile(ZipRunResultStorage.this.getPathFromId(runId), new LinkOption[0]);
            }
        });
        try {
            return future.get();
        }
        catch (InterruptedException | ExecutionException e) {
            this.logger.logf(Level.WARNING, e, "Checking result availability failed: %s", e.getMessage());
            return false;
        }
    }

    @Override
    public FileContent getResults(final String runId) throws IOException {
        Future<FileContent> resultFuture = this.executor.submit(new Callable<FileContent>(){

            @Override
            public FileContent call() throws Exception {
                Path zipFile = ZipRunResultStorage.this.getPathFromId(runId);
                ZipRunResultStorage.this.logger.logf(Level.FINEST, "Reading result zip %s.", zipFile);
                return FileUtils.getFileContent(zipFile);
            }
        });
        try {
            return resultFuture.get();
        }
        catch (InterruptedException | ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException)cause;
            }
            throw new IOException(e);
        }
    }

    private Path getPathFromId(String runId) {
        return this.folderPath.resolve(runId + ".zip");
    }

    @Override
    public boolean isResultStoringSupported() {
        return true;
    }
}

