/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.master.workerside.bootstrapping;

import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Strings;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.net.InetAddress;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.logging.Level;
import java.util.regex.Pattern;
import org.sosy_lab.verifiercloud.Modes;
import org.sosy_lab.verifiercloud.global.Constants;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.processes.ProcessExecutionException;
import org.sosy_lab.verifiercloud.global.processes.ProcessExecutor;
import org.sosy_lab.verifiercloud.global.statistics.StatisticsCollector;
import org.sosy_lab.verifiercloud.global.util.system.SSHCommandBuilder;
import org.sosy_lab.verifiercloud.global.util.system.SystemInformationUtils;
import org.sosy_lab.verifiercloud.master.workerside.bootstrapping.DefaultWorkerDispatcher;
import org.sosy_lab.verifiercloud.transportable.units.memory.MemoryUnit;
import org.sosy_lab.verifiercloud.transportable.units.time.TimeInterval;
import org.sosy_lab.verifiercloud.transportable.workerstart.WorkerStartInformation;

public class StartWorkerRunnable
implements Runnable {
    private static final ImmutableList<String> WORKER_JVM_PARAMETERS = ImmutableList.of("-XX:+HeapDumpOnOutOfMemoryError", "-Xmx256M");
    private static final String WORKER_LOG_FILENAME = "'worker-'yyyy-MM-dd'T'HHmmss'.log'";
    private static final TimeInterval MAXIMUM_TIMEOUT_TIME = TimeInterval.milliseconds(250L);
    private static final String WORKER_START_SCRIPT_PATH = "~/vcloud-start-worker.sh";
    private final String masterHost;
    private final Logger logger;
    private final StatisticsCollector statisticsCollector;
    private final String targetHost;
    private final String remoteWorkingDir;
    private final boolean enableUserLoginCheck;
    private final String workerId;
    private final int maxRunResultFiles;
    private final int workerToMasterPort;
    private final boolean useReverseTunnel;
    private final boolean tryHarder;
    private final boolean requirePingReachability;

    public StartWorkerRunnable(WorkerStartInformation workerStartInformation, String masterHost, String workerId, int maxRunResultFiles, boolean tryHarder, int workerToMasterPort, boolean requirePingReachability, StatisticsCollector statisticsCollector, Logger logger) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(masterHost));
        this.masterHost = masterHost;
        this.targetHost = Preconditions.checkNotNull(workerStartInformation.getHostname());
        this.workerId = Preconditions.checkNotNull(workerId);
        this.maxRunResultFiles = maxRunResultFiles;
        this.enableUserLoginCheck = workerStartInformation.enableUserLoginCheck();
        this.useReverseTunnel = workerStartInformation.useReverseTunnel();
        this.remoteWorkingDir = workerStartInformation.getPath().isPresent() ? workerStartInformation.getPath().get() : Constants.DEFAULT_WORKER_WORKING_FOLDER;
        this.workerToMasterPort = workerToMasterPort;
        this.requirePingReachability = requirePingReachability;
        this.logger = Preconditions.checkNotNull(logger);
        this.statisticsCollector = statisticsCollector;
        this.tryHarder = tryHarder;
    }

    @Override
    public void run() {
        String startWorkerCmd;
        this.logger.logf(Level.INFO, "Starting worker on %s.", this.targetHost);
        if (this.requirePingReachability && !StartWorkerRunnable.testHostReachability(this.targetHost)) {
            this.logger.logf(Level.WARNING, "Cannot find host %s.", this.targetHost);
            return;
        }
        ProcessExecutor pe = new ProcessExecutor(this.logger);
        String remoteCommand = String.format("mkdir -p %s; chmod go-rwx %s;", this.remoteWorkingDir, this.remoteWorkingDir);
        String sshCommand = SSHCommandBuilder.sshTo(this.targetHost, remoteCommand).build();
        try {
            pe.executeProcess(sshCommand);
        }
        catch (ProcessExecutionException e) {
            this.logger.logf(Level.WARNING, "Cannot connect to host %s.", this.targetHost);
            return;
        }
        if (this.tryHarder) {
            Optional<MemoryUnit> spaceLeft = this.spaceLeftOnTarget(pe);
            if (!spaceLeft.isPresent()) {
                this.logger.logf(Level.WARNING, "Could not determine available space on %s in %s.", this.targetHost, this.remoteWorkingDir);
            } else if (spaceLeft.get().isZero()) {
                Optional<MemoryUnit> freeAfterCleanup = this.cleanUpRemote(pe);
                if (freeAfterCleanup.isPresent()) {
                    MemoryUnit freeMemory = freeAfterCleanup.get();
                    if (freeMemory.isZero()) {
                        this.logger.logf(Level.WARNING, "No space available on %s in %s. Clean up manually!", this.targetHost, this.remoteWorkingDir);
                        return;
                    }
                    this.logger.logf(Level.INFO, "Cleaned up remote working directory and freed up %s.", new Object[0]);
                } else {
                    this.logger.logf(Level.WARNING, "Could not determine available space on %s in %s.", this.targetHost, this.remoteWorkingDir);
                    return;
                }
            }
        }
        String sep = StandardSystemProperty.FILE_SEPARATOR.value();
        String targetProgramLocation = this.remoteWorkingDir + sep + "worker.jar";
        Path programFile = SystemInformationUtils.getLocalProgramFile();
        if (Files.isDirectory(programFile, new LinkOption[0])) {
            this.logger.logf(Level.SEVERE, "Cannot spawn Workers from source code. Start jar-File instead.", new Object[0]);
            return;
        }
        Path localProgramPath = programFile;
        String copyCommand = "scp " + localProgramPath + " " + this.targetHost + ":" + targetProgramLocation;
        try {
            pe.executeProcess(copyCommand);
            this.logger.logf(Level.FINEST, "Copied program %s to host %s.", localProgramPath, this.targetHost);
        }
        catch (ProcessExecutionException e) {
            this.logger.logf(Level.WARNING, "Cannot copy program %s to host %s.", localProgramPath, this.targetHost);
            return;
        }
        try {
            startWorkerCmd = this.createStartCommand(targetProgramLocation, sep, pe);
        }
        catch (WorkerStartException e) {
            this.logger.logf(Level.WARNING, "Failed to start worker on %s: %s", this.targetHost, e.getMessage());
            return;
        }
        this.logger.logf(Level.ALL, "Executing command: %s", startWorkerCmd);
        try {
            pe.executeProcess(startWorkerCmd);
            this.logger.logf(Level.INFO, "Start command for worker %s terminated.", this.targetHost);
            this.statisticsCollector.increment("Number of worker starts");
        }
        catch (ProcessExecutionException e) {
            this.logger.logf(Level.WARNING, "Failed to start worker on %s.", this.targetHost);
        }
    }

    private Optional<MemoryUnit> cleanUpRemote(ProcessExecutor pe) {
        String remoteCommand = String.format("rm -r %1$s/%2$s* %1$s/%3$s", this.remoteWorkingDir, "tmp_", "files");
        String sshCommand = SSHCommandBuilder.sshTo(this.targetHost, remoteCommand).build();
        try {
            pe.executeProcess(sshCommand);
        }
        catch (ProcessExecutionException e) {
            this.logger.logf(Level.ALL, "Problem cleaning up space on host %s.", this.targetHost);
        }
        return this.spaceLeftOnTarget(pe);
    }

    private Optional<MemoryUnit> spaceLeftOnTarget(ProcessExecutor pe) {
        String result;
        String remoteDfCommand = String.format("df --portability '%s' | tail -n 1", this.remoteWorkingDir);
        String remoteFreeSpace = SSHCommandBuilder.sshTo(this.targetHost, remoteDfCommand).build();
        try {
            result = pe.executeProcess(remoteFreeSpace);
        }
        catch (ProcessExecutionException e) {
            return Optional.absent();
        }
        FluentIterable<String> dfTokens = FluentIterable.from(Splitter.on(Pattern.compile(" +")).split(result));
        if (dfTokens.size() != 6) {
            this.logger.logf(Level.FINE, "'%s' returned with unexpected output '%s'.", remoteDfCommand, result);
            return Optional.absent();
        }
        String availInByte = dfTokens.get(3);
        if (!availInByte.matches("[0-9]+")) {
            this.logger.logf(Level.FINE, "'%s' returned with unexpected output '%s' (%s is not a number).", remoteDfCommand, result, availInByte);
            return Optional.absent();
        }
        MemoryUnit avail = MemoryUnit.bytes(Long.parseLong(availInByte));
        return Optional.of(avail);
    }

    private String createStartCommand(String targetProgramLocation, String pathSeparator, ProcessExecutor pe) throws WorkerStartException {
        SimpleDateFormat logfilenameFormater = new SimpleDateFormat(WORKER_LOG_FILENAME);
        String logfileName = logfilenameFormater.format(new Date());
        String logTargetPath = this.remoteWorkingDir + pathSeparator + logfileName;
        String crashReportDir = this.remoteWorkingDir + pathSeparator + "crashReports";
        ImmutableList<String> workerParameters = ImmutableList.of(Modes.WORKER.getOptionName(), StartWorkerRunnable.formatParameter("master", this.useReverseTunnel ? "localhost" : this.masterHost), StartWorkerRunnable.formatParameter("master-port", this.workerToMasterPort), StartWorkerRunnable.formatParameter("worker-id", this.workerId), StartWorkerRunnable.formatParameter("stop-worker-on-userlogin", this.enableUserLoginCheck ? "true" : "false"), StartWorkerRunnable.formatParameter("max-run-result-files", this.maxRunResultFiles));
        String remoteCommand = String.format("%s %s -XX:HeapDumpPath=%s -jar %s %s &> %s 2>&1", this.getJavaVersion(pe), Joiner.on(' ').join(WORKER_JVM_PARAMETERS), crashReportDir, targetProgramLocation, Joiner.on(' ').join(workerParameters), logTargetPath);
        remoteCommand = String.format("nohup bash -c 'cd /;exec $( test -x %1$s && echo %1$s ) %2$s &' &> /dev/null &", WORKER_START_SCRIPT_PATH, remoteCommand);
        SSHCommandBuilder sshCommand = SSHCommandBuilder.sshTo(this.targetHost, remoteCommand);
        if (this.useReverseTunnel) {
            sshCommand.setUpReverseTunnel(this.workerToMasterPort, this.workerToMasterPort, DefaultWorkerDispatcher.MAX_WORKER_STARTUP_TIME);
        }
        return sshCommand.build();
    }

    private static String formatParameter(String parameterName, Object parameterValue) {
        return String.format("--%s %s", parameterName, parameterValue);
    }

    private String getJavaVersion(ProcessExecutor pe) throws WorkerStartException {
        String JAVA_VERSION = "1.7";
        String DEFAULT_JAVA = "java";
        String ALTERNATIVE_JAVA = "/usr/lib/jvm/java-7-openjdk-amd64/jre/bin/java";
        String versionCommand = SSHCommandBuilder.sshTo(this.targetHost, "java -version 2>&1 | head -n 1").build();
        try {
            String versionString = pe.executeProcess(versionCommand);
            return versionString.contains("1.7") ? "java" : "/usr/lib/jvm/java-7-openjdk-amd64/jre/bin/java";
        }
        catch (ProcessExecutionException e) {
            throw new WorkerStartException("Java version determination failure.");
        }
    }

    private static boolean testHostReachability(String host) {
        try {
            return InetAddress.getByName(host).isReachable((int)MAXIMUM_TIMEOUT_TIME.toMilliseconds());
        }
        catch (IOException e) {
            return false;
        }
    }

    private static class WorkerStartException
    extends Exception {
        private static final long serialVersionUID = 427035280755993858L;

        public WorkerStartException(String message) {
            super(Preconditions.checkNotNull(message));
        }
    }
}

