/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.transportable.filecontent;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.hash.HashCode;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import com.google.common.io.CountingOutputStream;
import com.google.common.io.FileBackedOutputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;
import javax.annotation.concurrent.Immutable;
import org.sosy_lab.verifiercloud.global.util.HashUtils;
import org.sosy_lab.verifiercloud.transportable.filecontent.AbstractFileContent;
import org.sosy_lab.verifiercloud.transportable.filecontent.CorruptedFileException;
import org.sosy_lab.verifiercloud.transportable.units.memory.MemoryUnit;

@Immutable
public class BigZipFileContent
extends AbstractFileContent {
    private static final long serialVersionUID = 1L;
    private static final String COMPRESSED_FILE_EXTENTION = ".zip";
    private static final int COMPRESSION_LEVEL = 9;
    private static final MemoryUnit fileThreshold = MemoryUnit.megabyte(25L);
    private final MemoryUnit fileSize;
    private final MemoryUnit representationSize;
    private final HashCode fileHash;
    private final transient boolean compressed;
    private transient Path content;
    private final transient AtomicBoolean tmpFile = new AtomicBoolean(false);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static BigZipFileContent fromPath(Path file, boolean lazyCompressing) throws IOException {
        Path conntent;
        MemoryUnit fileSize = MemoryUnit.bytes(Files.size(file));
        HashCode fileHash = HashUtils.hashFile(file);
        if (lazyCompressing) {
            conntent = Preconditions.checkNotNull(file);
        } else {
            Files.createDirectories(BigZipFileContent.getDeserialisationPath(), new FileAttribute[0]);
            conntent = BigZipFileContent.getUnambiguousZipFilePath(fileHash);
            Deflater deflater = new Deflater(9);
            try (OutputStream output = Files.newOutputStream(conntent, new OpenOption[0]);
                 DeflaterOutputStream compressOS = new DeflaterOutputStream(output, deflater);){
                Files.copy(file, compressOS);
            }
            finally {
                deflater.end();
            }
        }
        return new BigZipFileContent(conntent, fileHash, fileSize, !lazyCompressing);
    }

    public static BigZipFileContent fromRepresentation(Path file, boolean skipRehashing) throws IOException, CorruptedFileException {
        Preconditions.checkArgument(Files.isRegularFile(file, new LinkOption[0]), "%s is not a regular file.", file);
        List<String> tokens = BigZipFileContent.getFileNameTokens(file);
        Preconditions.checkArgument(2 <= tokens.size() && 3 >= tokens.size());
        boolean compressed = tokens.size() == 3;
        HashCode fileNameHash = HashCode.fromString(tokens.get(0));
        MemoryUnit representationNameSize = MemoryUnit.fromString(tokens.get(1));
        MemoryUnit fileNameSize = compressed ? MemoryUnit.fromString(tokens.get(2)) : representationNameSize;
        if (skipRehashing) {
            MemoryUnit representationSize = MemoryUnit.bytes(Files.size(file));
            if (!representationSize.equals(representationNameSize)) {
                throw new CorruptedFileException(file + " has invalid file size.");
            }
        } else {
            HashCode fileHash = compressed ? HashUtils.hashCompressedFile(file) : HashUtils.hashFile(file);
            if (!fileHash.equals(fileNameHash)) {
                throw new CorruptedFileException(file + " has invalid file hash.");
            }
        }
        return new BigZipFileContent(file, fileNameHash, fileNameSize, compressed);
    }

    private BigZipFileContent(Path compressedContent, HashCode fileHash, MemoryUnit fileSize, boolean compressed) throws IOException {
        this.content = compressedContent;
        this.fileSize = fileSize;
        this.fileHash = fileHash;
        this.compressed = compressed;
        this.representationSize = MemoryUnit.bytes(Files.size(compressedContent));
    }

    @Override
    public HashCode getFileHash() {
        return this.fileHash;
    }

    @Override
    public MemoryUnit getFileSize() {
        return this.fileSize;
    }

    @Override
    public MemoryUnit getRepresentationSize() {
        return this.representationSize;
    }

    @Override
    public void writeToPath(Path targetPath) throws IOException {
        if (this.compressed) {
            try (InputStream decompressIS = this.createInputStream();){
                Files.copy(decompressIS, targetPath, new CopyOption[0]);
            }
        } else {
            this.moveOrCopy(targetPath);
        }
    }

    @Override
    public Path writeRepresentationToPath(Path targetDir) throws IOException {
        Preconditions.checkArgument(Files.isDirectory(targetDir, new LinkOption[0]));
        String fileName = this.compressed ? String.format("%s_%s_%s.zip", this.fileHash, Files.size(this.content), this.fileSize.toByte()) : String.format("%s_%s", this.fileHash, Files.size(this.content));
        Path file = targetDir.resolve(fileName);
        this.moveOrCopy(file);
        return file;
    }

    private void moveOrCopy(Path targetPath) throws IOException {
        if (this.tmpFile.getAndSet(false)) {
            Files.move(this.content, targetPath, new CopyOption[0]);
            this.content = targetPath;
        } else {
            try {
                Files.createLink(targetPath, this.content);
            }
            catch (IOException e) {
                Files.copy(this.content, targetPath, new CopyOption[0]);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public String getContent() {
        if (this.compressed) {
            try (InputStream decompressIS = this.createInputStream();){
                byte[] content = ByteStreams.toByteArray(decompressIS);
                String string = new String(content, Charsets.UTF_8);
                return string;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        try {
            return new String(Files.readAllBytes(this.content), Charsets.UTF_8);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public ByteSource getContentAsByteSource() {
        return new ByteSource(){

            @Override
            public InputStream openStream() throws IOException {
                return BigZipFileContent.this.createInputStream();
            }
        };
    }

    private InputStream createInputStream() throws IOException {
        InputStream fileStream = Files.newInputStream(this.content, new OpenOption[0]);
        if (this.compressed) {
            return new InflaterInputStream(fileStream);
        }
        return fileStream;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeObject(ObjectOutputStream oos) throws IOException {
        oos.writeObject(this.fileHash);
        oos.writeObject(this.fileSize);
        if (this.compressed) {
            oos.writeObject(this.representationSize);
            Files.copy(this.content, oos);
        } else {
            try (FileBackedOutputStream fileBackedOut = new FileBackedOutputStream((int)fileThreshold.toByte());){
                try (CountingOutputStream output = new CountingOutputStream(fileBackedOut);
                     BufferedOutputStream outputBuffer = new BufferedOutputStream(output);){
                    Deflater deflater = new Deflater(9);
                    try (DeflaterOutputStream compressOS = new DeflaterOutputStream((OutputStream)outputBuffer, deflater, 8192);){
                        Files.copy(this.content, compressOS);
                        compressOS.flush();
                    }
                    finally {
                        deflater.end();
                    }
                    MemoryUnit representationSize = MemoryUnit.bytes(output.getCount());
                    oos.writeObject(representationSize);
                    fileBackedOut.asByteSource().copyTo(oos);
                }
                finally {
                    fileBackedOut.reset();
                }
            }
        }
    }

    private void readObject(ObjectInputStream ois) throws IOException {
        MemoryUnit representationSize;
        MemoryUnit fileSize;
        HashCode fileHash;
        try {
            fileHash = (HashCode)ois.readObject();
            fileSize = (MemoryUnit)ois.readObject();
            representationSize = (MemoryUnit)ois.readObject();
        }
        catch (ClassCastException | ClassNotFoundException e) {
            throw new IOException(e);
        }
        Files.createDirectories(BigZipFileContent.getDeserialisationPath(), new FileAttribute[0]);
        Path compressedContent = BigZipFileContent.getUnambiguousZipFilePath(fileHash);
        try {
            Field fileHashField = this.getClass().getDeclaredField("fileHash");
            fileHashField.setAccessible(true);
            fileHashField.set(this, fileHash);
            Field fileSizeField = this.getClass().getDeclaredField("fileSize");
            fileSizeField.setAccessible(true);
            fileSizeField.set(this, fileSize);
            Field contentField = this.getClass().getDeclaredField("content");
            contentField.setAccessible(true);
            contentField.set(this, compressedContent);
            Field representationSizeField = this.getClass().getDeclaredField("representationSize");
            representationSizeField.setAccessible(true);
            representationSizeField.set(this, representationSize);
            Field compressedField = this.getClass().getDeclaredField("compressed");
            compressedField.setAccessible(true);
            compressedField.set(this, true);
            Field tmpFileField = this.getClass().getDeclaredField("tmpFile");
            tmpFileField.setAccessible(true);
            tmpFileField.set(this, new AtomicBoolean(true));
        }
        catch (IllegalAccessException | IllegalArgumentException | NoSuchFieldException | SecurityException e) {
            throw new IOException(e);
        }
        InputStream contentStream = ByteStreams.limit(ois, representationSize.toByte());
        Files.copy(contentStream, compressedContent, new CopyOption[0]);
    }

    private static Path getUnambiguousZipFilePath(HashCode fileNamePrefix) {
        Path dir = BigZipFileContent.getDeserialisationPath();
        String unambiguousFileName = fileNamePrefix.toString() + UUID.randomUUID();
        Path unambiguousFilePath = dir.resolve(unambiguousFileName + COMPRESSED_FILE_EXTENTION);
        while (Files.exists(unambiguousFilePath, new LinkOption[0])) {
            unambiguousFileName = unambiguousFileName + "_";
            unambiguousFilePath = dir.resolve(unambiguousFileName + COMPRESSED_FILE_EXTENTION);
        }
        return unambiguousFilePath;
    }
}

