/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.transportable.info.processors;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.ParseException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import org.sosy_lab.verifiercloud.global.util.system.SystemEnvironmentException;
import org.sosy_lab.verifiercloud.transportable.info.processors.Processor;

public final class ProcessorParser {
    private static final Path CPUINFO = Paths.get("/proc/cpuinfo", new String[0]);
    private static final String PROCESSOR_ID_KEY = "processor";
    private static final String SOCKET_ID_KEY = "physical id";
    private static final String CORE_ID_KEY = "core id";

    private ProcessorParser() {
    }

    static Set<Processor> parseProcessorInformation(Iterable<String> source) throws ParseException {
        HashSet<Processor> procList = Sets.newHashSet();
        int blockCount = 0;
        int lineNumber = 1;
        Optional<Integer> processorId = Optional.absent();
        Optional<Integer> coreId = Optional.absent();
        Optional<Integer> socketId = Optional.absent();
        for (String line : source) {
            Optional<KeyValuePair> parsedLine = ProcessorParser.parseLine(line, lineNumber++);
            if (parsedLine.isPresent()) {
                KeyValuePair keyValuePair = parsedLine.get();
                String key = keyValuePair.getKey();
                String value = keyValuePair.getValue();
                switch (key) {
                    case "processor": {
                        processorId = ProcessorParser.getIntegerValue(processorId, lineNumber, line, value);
                        break;
                    }
                    case "physical id": {
                        socketId = ProcessorParser.getIntegerValue(socketId, lineNumber, line, value);
                        break;
                    }
                    case "core id": {
                        coreId = ProcessorParser.getIntegerValue(coreId, lineNumber, line, value);
                        break;
                    }
                }
                continue;
            }
            if (!processorId.isPresent()) {
                throw new ParseException(String.format("Current block does not contain a processor id (%s).", PROCESSOR_ID_KEY), lineNumber);
            }
            procList.add(new Processor((Integer)processorId.get(), coreId.isPresent() ? (Integer)coreId.get() : blockCount, socketId.isPresent() ? (Integer)socketId.get() : 0));
            processorId = Optional.absent();
            coreId = Optional.absent();
            socketId = Optional.absent();
            ++blockCount;
        }
        Preconditions.checkState(!processorId.isPresent());
        Preconditions.checkState(!socketId.isPresent());
        Preconditions.checkState(!coreId.isPresent());
        return procList;
    }

    public static Set<Processor> parseProcessorInformation() throws SystemEnvironmentException {
        try {
            return ProcessorParser.parseProcessorInformation(Files.readAllLines(CPUINFO, Charset.defaultCharset()));
        }
        catch (IOException | ParseException e) {
            throw new SystemEnvironmentException(e);
        }
    }

    private static Optional<Integer> getIntegerValue(Optional<Integer> priorValue, int lineNumber, String line, String value) throws ParseException {
        if (priorValue.isPresent() || value.isEmpty()) {
            throw new ParseException("Key without value or key duplicate: " + line, lineNumber);
        }
        try {
            return Optional.of(Integer.valueOf(value));
        }
        catch (NumberFormatException e) {
            throw new ParseException("Expected integer value: " + line, lineNumber);
        }
    }

    private static Optional<KeyValuePair> parseLine(String line, int lineNumber) throws ParseException {
        Preconditions.checkArgument(!line.contains("\n"));
        Preconditions.checkArgument(lineNumber >= 0);
        if (line.trim().isEmpty()) {
            return Optional.absent();
        }
        Iterable<String> tokens = Splitter.on(':').split(line);
        if (FluentIterable.from(tokens).size() != 2) {
            throw new ParseException("Line in " + CPUINFO + " cannot be parsed.", lineNumber);
        }
        Iterator<String> it = tokens.iterator();
        String key = it.next().trim();
        String value = it.next().trim();
        if (key.isEmpty()) {
            throw new RuntimeException("Line " + lineNumber + ": '" + line + "' contains no key.");
        }
        return Optional.of(new KeyValuePair(key, value));
    }

    private static class KeyValuePair {
        private final String key;
        private final String value;

        public KeyValuePair(String key, String value) {
            Preconditions.checkArgument(!Strings.isNullOrEmpty(key));
            this.key = key;
            Preconditions.checkNotNull(value);
            this.value = value;
        }

        public String getKey() {
            return this.key;
        }

        public String getValue() {
            return this.value;
        }
    }
}

