/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.worker;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.inject.Inject;
import com.google.inject.name.Named;
import java.io.IOException;
import java.net.UnknownHostException;
import java.nio.file.Path;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.global.application.ApplicationService;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.util.deadlocks.DeadlockDetection;
import org.sosy_lab.verifiercloud.global.util.disk_space.NoFreeDiskSpaceDetection;
import org.sosy_lab.verifiercloud.global.util.error_trigger.ErrorTriggerExecutor;
import org.sosy_lab.verifiercloud.global.util.system.SystemInformationProvider;
import org.sosy_lab.verifiercloud.transportable.units.memory.MemoryUnit;
import org.sosy_lab.verifiercloud.transportable.units.time.TimeInterval;
import org.sosy_lab.verifiercloud.worker.files.WorkerFileStorage;
import org.sosy_lab.verifiercloud.worker.info.WorkerCrashReportSender;
import org.sosy_lab.verifiercloud.worker.maintenance.WorkingDirectoryCleanup;
import org.sosy_lab.verifiercloud.worker.network.MasterConnection;
import org.sosy_lab.verifiercloud.worker.run.WorkerModel;
import org.sosy_lab.verifiercloud.worker.run.WorkerStateListener;

public class Worker
implements ApplicationService {
    private final TimeInterval MAXTIME_WAIT_FOR_CONNECTION = TimeInterval.seconds(30L);
    private final Logger logger;
    private final WorkerCrashReportSender exceptionMessageSender;
    private final Thread.UncaughtExceptionHandler uncaughtExceptionHandler;
    private final ErrorTriggerExecutor errorTriggerExecutor;
    private final MasterConnection masterConnection;
    private final WorkerStateListener workerStateListener;
    private final WorkerFileStorage files;
    private final WorkerModel workerModel;
    private final Path workingDir;
    private final SystemInformationProvider systemInfoProvider;
    private final NoFreeDiskSpaceDetection noFreeDiskSpaceDetection;
    private final AtomicBoolean workerStoppedFlag = new AtomicBoolean(false);

    @Inject
    public Worker(MasterConnection masterConnection, WorkerStateListener workerStateListener, WorkerFileStorage workerFileStorage, SystemInformationProvider systemInfoProvider, WorkerModel workerModel, @Named(value="workingDir") Path workingDir, WorkerCrashReportSender exceptionMessageSender, NoFreeDiskSpaceDetection noFreeDiskSpaceDetection, Thread.UncaughtExceptionHandler uncaughtExceptionHandler, ErrorTriggerExecutor errorTriggerExecutor, Logger logger) {
        this.logger = Preconditions.checkNotNull(logger);
        this.exceptionMessageSender = Preconditions.checkNotNull(exceptionMessageSender);
        this.uncaughtExceptionHandler = Preconditions.checkNotNull(uncaughtExceptionHandler);
        this.errorTriggerExecutor = Preconditions.checkNotNull(errorTriggerExecutor);
        this.masterConnection = Preconditions.checkNotNull(masterConnection);
        this.workerStateListener = Preconditions.checkNotNull(workerStateListener);
        this.workerModel = Preconditions.checkNotNull(workerModel);
        this.files = Preconditions.checkNotNull(workerFileStorage);
        this.workingDir = Preconditions.checkNotNull(workingDir);
        this.systemInfoProvider = Preconditions.checkNotNull(systemInfoProvider);
        this.noFreeDiskSpaceDetection = Preconditions.checkNotNull(noFreeDiskSpaceDetection);
    }

    @Override
    public void start() {
        DeadlockDetection deadlockDetection = new DeadlockDetection(this.errorTriggerExecutor, this.logger);
        deadlockDetection.start();
        Thread.setDefaultUncaughtExceptionHandler(this.uncaughtExceptionHandler);
        Thread.currentThread().setUncaughtExceptionHandler(this.uncaughtExceptionHandler);
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable(){

            @Override
            public void run() {
                Worker.this.stop();
            }
        }));
        try {
            this.logger.logf(Level.INFO, "Starting worker on %s.", this.systemInfoProvider.getHostname());
        }
        catch (UnknownHostException e) {
            this.logger.logf(Level.SEVERE, "Starting worker failed. Could not determine hostname.", new Object[0]);
            this.stop(e);
            return;
        }
        this.workerModel.registerStateChangeListener(this.workerStateListener);
        try {
            this.files.initializeFileStorage(this.workingDir);
        }
        catch (IOException e) {
            this.logger.logf(Level.SEVERE, e, "Cannot initialize file storage: %s.", e.getMessage());
            this.stop(e);
            return;
        }
        ListenableFuture<Boolean> connectionAttempt = this.masterConnection.connectToMaster();
        try {
            boolean successOfConnection = (Boolean)connectionAttempt.get(this.MAXTIME_WAIT_FOR_CONNECTION.toMilliseconds(), TimeUnit.MILLISECONDS);
            if (!successOfConnection) {
                this.logger.logf(Level.SEVERE, "Connection to master failed.", new Object[0]);
                this.stop();
                return;
            }
            this.logger.logf(Level.INFO, "Connection to master established.", new Object[0]);
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            this.logger.logf(Level.SEVERE, e, "Exception when connecting: %s", e.getMessage());
            this.stop();
            return;
        }
        try {
            MemoryUnit deltedFiles = new WorkingDirectoryCleanup(this.workingDir, this.logger).cleanUpOldFiles();
            this.logger.logf(Level.INFO, "Removed %s of old data.", deltedFiles);
        }
        catch (IOException e) {
            this.logger.logf(Level.SEVERE, e, "IOException during initial cleanup: %s", e.getMessage());
            this.stop(e);
            return;
        }
        try {
            this.files.indexFiles();
        }
        catch (IOException e) {
            this.logger.logf(Level.SEVERE, e, "Cannot index file storage: %s.", e.getMessage());
            this.stop(e);
            return;
        }
        this.exceptionMessageSender.sendAllCrashReportsToMaster();
        this.noFreeDiskSpaceDetection.start();
        this.workerModel.finishInitialization();
    }

    @Override
    public void stop() {
        if (this.workerStoppedFlag.getAndSet(true)) {
            return;
        }
        this.logger.logf(Level.INFO, "Stopping worker.", new Object[0]);
        this.noFreeDiskSpaceDetection.stop();
        this.workerModel.stop();
        this.masterConnection.stopConnection();
    }

    public void stop(Throwable e) {
        this.exceptionMessageSender.addCrashReport(e);
        this.errorTriggerExecutor.executeErrorTrigger(e, "Stopping worker caused by:");
        this.stop();
    }
}

