/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.worker.info;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.Inject;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.ref.PhantomReference;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ThreadFactory;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.util.FileUtils;
import org.sosy_lab.verifiercloud.global.util.system.SystemInformationProvider;
import org.sosy_lab.verifiercloud.global.util.system.SystemInformationUtils;
import org.sosy_lab.verifiercloud.transportable.filecontent.FileContent;
import org.sosy_lab.verifiercloud.transportable.units.time.TimeInterval;
import org.sosy_lab.verifiercloud.worker.network.MasterConnection;

public class WorkerCrashReportSender {
    public static final String CRASH_REPORT_DIR_NAME = "crashReports";
    private static final Path CRASH_REPORT_PATH = SystemInformationUtils.getWorkingDir().resolve("crashReports");
    private final SimpleDateFormat timeFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
    private final MasterConnection masterConnection;
    private final SystemInformationProvider systemInformationProvider;
    private final ThreadFactory threadFactory;
    private final Logger logger;
    private final ReferenceQueue<FileContent> crashReportsToDelete;
    private final Map<Reference<FileContent>, Path> pathsOfCrashReports;

    @Inject
    public WorkerCrashReportSender(MasterConnection masterConnection, SystemInformationProvider systemInformationProvider, Logger logger) {
        this.logger = Preconditions.checkNotNull(logger);
        this.systemInformationProvider = Preconditions.checkNotNull(systemInformationProvider);
        this.masterConnection = masterConnection;
        this.threadFactory = new ThreadFactoryBuilder().setNameFormat(this.getClass().getSimpleName() + "-%d").build();
        this.crashReportsToDelete = new ReferenceQueue();
        this.pathsOfCrashReports = Maps.newHashMap();
    }

    public void addCrashReport(Throwable t) {
        String uniqueName = UUID.randomUUID().toString();
        Path messageFile = CRASH_REPORT_PATH.resolve(uniqueName);
        try {
            Files.createDirectories(CRASH_REPORT_PATH, new FileAttribute[0]);
        }
        catch (IOException ioException) {
            this.logger.logf(Level.WARNING, ioException, "IOException while creating directory for error reports: %s", ioException.getMessage());
            return;
        }
        try (BufferedWriter fileWriter = Files.newBufferedWriter(messageFile, Charsets.UTF_8, StandardOpenOption.CREATE_NEW);
             PrintWriter printWriter = new PrintWriter(fileWriter);){
            printWriter.print("Worker " + this.systemInformationProvider.getHostname() + "crashed at");
            printWriter.println(this.timeFormat.format(new Date()));
            printWriter.println(t.getClass().getName());
            printWriter.println(t.getMessage());
            printWriter.println();
            t.printStackTrace(printWriter);
            printWriter.println();
        }
        catch (IOException ioException) {
            this.logger.logf(Level.WARNING, ioException, "IOException while writting error information: %s", ioException.getMessage());
        }
    }

    public void sendAllCrashReportsToMaster() {
        this.threadFactory.newThread(new CrashReportSender()).start();
    }

    private static class OnlyFilesFilter
    implements DirectoryStream.Filter<Path> {
        private OnlyFilesFilter() {
        }

        @Override
        public boolean accept(Path entry) throws IOException {
            return Files.isRegularFile(entry, new LinkOption[0]);
        }
    }

    private class CrashReportSender
    implements Runnable {
        private CrashReportSender() {
        }

        @Override
        public void run() {
            try {
                Files.createDirectories(CRASH_REPORT_PATH, new FileAttribute[0]);
                this.sendFiles();
                this.removeSentFiles();
            }
            catch (IOException e) {
                WorkerCrashReportSender.this.logger.logf(Level.WARNING, e, "Cannot read error messages: %s", e.getMessage());
            }
        }

        private void sendFiles() throws IOException {
            DirectoryStream<Path> files = Files.newDirectoryStream(CRASH_REPORT_PATH, new OnlyFilesFilter());
            for (Path crashReportFile : files) {
                FileContent content = FileUtils.getFileContent(crashReportFile);
                WorkerCrashReportSender.this.masterConnection.reportWorkerCrash(content);
                PhantomReference<FileContent> reference = new PhantomReference<FileContent>(content, WorkerCrashReportSender.this.crashReportsToDelete);
                WorkerCrashReportSender.this.pathsOfCrashReports.put(reference, crashReportFile);
            }
        }

        private void removeSentFiles() throws IOException {
            boolean finished = false;
            while (!finished) {
                try {
                    Reference reference = WorkerCrashReportSender.this.crashReportsToDelete.remove(TimeInterval.minutes(5L).toMilliseconds());
                    if (reference != null) {
                        Path crashReportFile = (Path)WorkerCrashReportSender.this.pathsOfCrashReports.get(reference);
                        Files.delete(crashReportFile);
                        continue;
                    }
                    finished = true;
                }
                catch (InterruptedException e) {
                    finished = true;
                }
            }
        }
    }
}

