/*
 * Decompiled with CFR 0.152.
 */
package org.sosy_lab.verifiercloud.worker.run.program_environment;

import com.google.common.base.Charsets;
import com.google.common.base.Functions;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.hash.HashCode;
import com.google.common.io.ByteSource;
import com.google.common.io.Resources;
import com.google.common.util.concurrent.CheckedFuture;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import org.sosy_lab.verifiercloud.global.file_storage.FileNotAvailableException;
import org.sosy_lab.verifiercloud.global.logging.Logger;
import org.sosy_lab.verifiercloud.global.util.DirDeleter;
import org.sosy_lab.verifiercloud.global.util.FileUtils;
import org.sosy_lab.verifiercloud.transportable.file_hierarchy.FileAtRelativePath;
import org.sosy_lab.verifiercloud.transportable.filecontent.FileContent;
import org.sosy_lab.verifiercloud.transportable.info.processors.NumaProcessorParser;
import org.sosy_lab.verifiercloud.transportable.info.processors.Processor;
import org.sosy_lab.verifiercloud.transportable.info.processors.ProcessorToProcessorIdFunction;
import org.sosy_lab.verifiercloud.transportable.run.Run;
import org.sosy_lab.verifiercloud.transportable.units.memory.MemoryUnit;
import org.sosy_lab.verifiercloud.transportable.units.time.TimeInterval;
import org.sosy_lab.verifiercloud.worker.files.WorkerFileStorage;
import org.sosy_lab.verifiercloud.worker.run.program_environment.DiffFileLister;
import org.sosy_lab.verifiercloud.worker.run.program_environment.TooManyResultFilesException;

public class ProgramEnvironment {
    public static final String RUN_WORKING_DIR_PREFIX = "tmp_";
    public static final String RUN_TEMP_DIR_POSTFIX = "_tempdir";
    public static final Iterable<String> TEMP_DIR_SHELL_VARIABLES = ImmutableList.of("TEMP", "TMP", "TEMPDIR", "TMPDIR");
    private static final boolean DEBUG_RUNEXECUTOR = "true".equalsIgnoreCase(System.getenv("VCLOUD_DEBUG"));
    private static final String STANDARD_ERROR = "verifiercloud.error";
    private static final String STANDARD_OUT = "verifiercloud.out";
    private static final String EXECUTOR_PATH = "verifiercloud_runexecutor.py";
    private static final String EXECUTOR_NAME = "runexecutor.py";
    private static final ImmutableCollection<String> EXECUTOR_FILES = ImmutableList.of("__init__.py", "cgroups.py", "runexecutor.py", "util.py", "oomhandler.py");
    private static final Path BENCHMARK_PATH = Paths.get("benchmark", new String[0]);
    private static final Splitter.MapSplitter RUN_INFO_SPLITTER = Splitter.on('\n').trimResults().omitEmptyStrings().withKeyValueSeparator("=");
    private static final TimeInterval FILE_TIMEOUT = TimeInterval.minutes(30L);
    private static final boolean isNumaSupportAvailable = NumaProcessorParser.isNUMAsupportAvailable();
    private final Map<Path, ByteSource> executorFiles;
    private final Logger logger;
    private final WorkerFileStorage workerFileStorage;
    private final Run run;
    private final boolean suppressCleanup;
    private final int maxRunResultFiles;
    private Optional<Path> workingDir;
    private Optional<Path> temporaryDir;
    private Optional<Path> systemStdout;
    private Optional<Path> systemStderr;
    private Path executionDirectory;
    private Path workerResultFile;
    private Optional<Iterable<FileAtRelativePath>> newFiles;
    private final ImmutableMap<Integer, MemoryUnit> memoryLimit;
    private final ImmutableSet<Processor> processorsToUse;

    public ProgramEnvironment(Logger logger, Run run, WorkerFileStorage workerFileStorage, ImmutableMap<Integer, MemoryUnit> reservedMemory, Set<Processor> processorsToUse, boolean suppressCleanup, int maxRunResultFiles) {
        this.logger = Preconditions.checkNotNull(logger);
        this.workerFileStorage = Preconditions.checkNotNull(workerFileStorage);
        this.run = Preconditions.checkNotNull(run);
        this.workingDir = Optional.absent();
        this.temporaryDir = Optional.absent();
        this.systemStdout = Optional.absent();
        this.systemStderr = Optional.absent();
        this.newFiles = Optional.absent();
        this.memoryLimit = Preconditions.checkNotNull(reservedMemory);
        this.processorsToUse = ImmutableSet.copyOf(processorsToUse);
        this.executorFiles = Maps.newHashMap();
        this.suppressCleanup = suppressCleanup;
        this.maxRunResultFiles = maxRunResultFiles;
        Path runExecutorFile = Paths.get(EXECUTOR_PATH, new String[0]);
        ByteSource runExecutor = Resources.asByteSource(ClassLoader.getSystemResource(EXECUTOR_NAME));
        this.executorFiles.put(runExecutorFile, runExecutor);
        for (String pathString : EXECUTOR_FILES) {
            Path tragetPath = BENCHMARK_PATH.resolve(pathString);
            ByteSource excutorFile = Resources.asByteSource(ClassLoader.getSystemResource(BENCHMARK_PATH + "/" + pathString));
            this.executorFiles.put(tragetPath, excutorFile);
        }
    }

    private String formatProcessorsToUse(Set<Processor> processorsToUse) {
        Preconditions.checkArgument(!processorsToUse.isEmpty(), "No processors assigned.");
        FluentIterable<String> processorIds = FluentIterable.from(processorsToUse).transform(ProcessorToProcessorIdFunction.INSTANCE).transform(Functions.toStringFunction());
        String processorString = Joiner.on(',').join(processorIds);
        return processorString;
    }

    public synchronized void createProgramEnviroment(Path baseDir) throws FileNotAvailableException, IOException {
        Preconditions.checkState(!this.workingDir.isPresent(), "Program enviroment for %s is already created.", baseDir);
        Preconditions.checkArgument(Files.isDirectory(baseDir, new LinkOption[0]), "%s is not an directory.", baseDir);
        Path programWorkingDir = this.getUniquePathFromPrefix(baseDir, RUN_WORKING_DIR_PREFIX + this.run.hashCode());
        Files.createDirectories(programWorkingDir, new FileAttribute[0]);
        this.logger.logf(Level.FINER, "Created directory %s for %s", programWorkingDir, this.run);
        this.workingDir = Optional.of(programWorkingDir);
        String suggestedTempDirName = programWorkingDir.getFileName().toString() + RUN_TEMP_DIR_POSTFIX;
        Path temporaryDir = this.getUniquePathFromPrefix(baseDir, suggestedTempDirName).toAbsolutePath();
        Files.createDirectories(temporaryDir, new FileAttribute[0]);
        this.logger.logf(Level.FINER, "Created temporary directory %s for %s.", temporaryDir, this.run);
        this.temporaryDir = Optional.of(temporaryDir);
        this.systemStdout = Optional.of(programWorkingDir.resolve(STANDARD_OUT));
        this.systemStderr = Optional.of(programWorkingDir.resolve(STANDARD_ERROR));
        this.linkFilesToWorkingDirectory(programWorkingDir);
        String relativeExecutionDir = this.run.getRelativeProgramWorkingDirectory().getRelativePath();
        this.executionDirectory = programWorkingDir.resolve(relativeExecutionDir);
        Files.createDirectories(this.executionDirectory, new FileAttribute[0]);
        this.workerResultFile = programWorkingDir.resolve(this.run.getWorkerResultFile().getRelativePath());
        Path benchmarkPath = programWorkingDir.resolve(BENCHMARK_PATH);
        Files.createDirectories(benchmarkPath, new FileAttribute[0]);
        for (Path path : this.executorFiles.keySet()) {
            Path targetPath = programWorkingDir.resolve(path);
            InputStream source = this.executorFiles.get(path).openStream();
            Throwable throwable = null;
            try {
                Files.copy(source, targetPath, new CopyOption[0]);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (source == null) continue;
                if (throwable != null) {
                    try {
                        source.close();
                    }
                    catch (Throwable x2) {
                        throwable.addSuppressed(x2);
                    }
                    continue;
                }
                source.close();
            }
        }
    }

    public synchronized String[] buildExecutorCommand() {
        ArrayList<String> command = Lists.newArrayList();
        command.add("python");
        command.add(EXECUTOR_PATH);
        if (DEBUG_RUNEXECUTOR) {
            command.add("--debug");
        } else {
            command.add("--quiet");
        }
        command.add("--dir");
        command.add(this.executionDirectory.toString());
        command.add("--output");
        command.add(this.workerResultFile.toString());
        long timeLimit = this.run.getLimitations().getTimeLimit().toSeconds();
        command.add("--timelimit");
        command.add(Long.toString(timeLimit));
        long memLimit = MemoryUnit.sum(this.memoryLimit.values()).toByte();
        command.add("--memlimit");
        command.add(Long.toString(memLimit));
        if (isNumaSupportAvailable) {
            Set<Integer> numaNodesToUse = Maps.filterValues(this.memoryLimit, MemoryIsNotZeroPredicate.INSTANCE).keySet();
            command.add("--memoryNodes");
            command.add(Joiner.on(',').join(numaNodesToUse));
        } else {
            Preconditions.checkArgument(this.memoryLimit.size() == 1);
            Preconditions.checkArgument(this.memoryLimit.containsKey(0));
        }
        Preconditions.checkState(!this.processorsToUse.isEmpty(), "No processors available.");
        command.add("--cores");
        command.add(this.formatProcessorsToUse(this.processorsToUse));
        command.add("--");
        command.addAll(this.run.getCommand());
        String[] result = new String[command.size()];
        result = command.toArray(result);
        this.logger.logf(Level.FINE, "cmdline for python-wrapper: '%s'", command);
        return result;
    }

    private Path getUniquePathFromPrefix(Path parent, String name) {
        Path uniquePath = parent.resolve(name);
        while (Files.exists(uniquePath, new LinkOption[0])) {
            uniquePath = parent.resolve(name + "_" + System.nanoTime());
        }
        return uniquePath;
    }

    private void linkFilesToWorkingDirectory(Path workingDir) throws FileNotAvailableException, IOException {
        Map<CheckedFuture<Path, FileNotAvailableException>, FileAtRelativePath> futureFiles = this.getFileFutures();
        this.linkFilesToDirectory(workingDir, futureFiles);
    }

    private void linkFilesToDirectory(Path workingDir, Map<CheckedFuture<Path, FileNotAvailableException>, FileAtRelativePath> futureFiles) throws FileNotAvailableException, IOException {
        for (Map.Entry<CheckedFuture<Path, FileNotAvailableException>, FileAtRelativePath> entry : futureFiles.entrySet()) {
            Path pathOfFile;
            CheckedFuture<Path, FileNotAvailableException> futureFile = entry.getKey();
            FileAtRelativePath fileAtRelativePath = entry.getValue();
            try {
                pathOfFile = futureFile.checkedGet(FILE_TIMEOUT.toSeconds(), TimeUnit.SECONDS);
            }
            catch (TimeoutException e) {
                throw new FileNotAvailableException(fileAtRelativePath.getFileHash(), (Throwable)e);
            }
            String relativeTarget = fileAtRelativePath.getRelativePathString();
            Path relativeTargetPath = workingDir.resolve(relativeTarget);
            Files.createDirectories(relativeTargetPath.getParent(), new FileAttribute[0]);
            Files.createLink(relativeTargetPath, pathOfFile);
            this.logger.logf(Level.FINEST, "Linked %s to path %s.", pathOfFile.getFileName(), relativeTargetPath);
        }
        this.logger.logf(Level.FINER, "Finished linking files for %s to %s", this.run, workingDir);
    }

    private Map<CheckedFuture<Path, FileNotAvailableException>, FileAtRelativePath> getFileFutures() {
        HashMap<CheckedFuture<Path, FileNotAvailableException>, FileAtRelativePath> futureFiles = Maps.newHashMap();
        for (FileAtRelativePath fileAtRelativePath : this.run.getFileHierarchy()) {
            HashCode fileHashCode = fileAtRelativePath.getFileHash();
            CheckedFuture<Path, FileNotAvailableException> futureFile = this.workerFileStorage.getFile(fileHashCode);
            futureFiles.put(futureFile, fileAtRelativePath);
        }
        return futureFiles;
    }

    public synchronized Path getWorkingDir() {
        Preconditions.checkState(this.workingDir.isPresent(), "Program environment cleaned up or not created.");
        return this.workingDir.get();
    }

    public synchronized Path getStdoutFile() {
        Preconditions.checkState(this.workingDir.isPresent(), "Program environment cleaned up or not created.");
        return this.systemStdout.get();
    }

    public synchronized Path getStderrFile() {
        Preconditions.checkState(this.workingDir.isPresent(), "Program environment cleaned up or not created.");
        return this.systemStderr.get();
    }

    public synchronized void updateEnvironment(Map<String, String> environment) {
        Preconditions.checkState(this.temporaryDir.isPresent(), "Program environment cleaned up or not created.");
        for (String variable : TEMP_DIR_SHELL_VARIABLES) {
            environment.put(variable, this.temporaryDir.get().toString());
        }
    }

    public synchronized Iterable<FileAtRelativePath> getNewFiles() throws IOException, TooManyResultFilesException {
        Preconditions.checkState(this.workingDir.isPresent());
        if (this.newFiles.isPresent()) {
            return this.newFiles.get();
        }
        HashSet<Path> knownRelativePaths = Sets.newHashSet();
        ImmutableList<Optional<Path>> metaFiles = ImmutableList.of(this.systemStdout, this.systemStderr);
        Path absolutePathWorkingDir = this.getAbsoluteWorkingDirPath();
        for (Optional optional : metaFiles) {
            if (!optional.isPresent()) continue;
            Path absolutePath = ((Path)optional.get()).toAbsolutePath();
            Path relativePath = absolutePathWorkingDir.relativize(absolutePath);
            knownRelativePaths.add(relativePath);
        }
        for (Path path : this.executorFiles.keySet()) {
            knownRelativePaths.add(path);
        }
        for (FileAtRelativePath fileAtRelativePath : this.run.getFileHierarchy()) {
            knownRelativePaths.add(Paths.get(fileAtRelativePath.getRelativePathString(), new String[0]));
        }
        DiffFileLister diffFileLister = new DiffFileLister(this.getWorkingDir(), knownRelativePaths, this.maxRunResultFiles);
        this.newFiles = Optional.of(diffFileLister.getNewFiles());
        return this.newFiles.get();
    }

    private Path getAbsoluteWorkingDirPath() {
        Preconditions.checkState(this.workingDir.isPresent(), "No working directory exists.");
        Path absolutePathWorkingDir = this.workingDir.get().toAbsolutePath();
        return absolutePathWorkingDir;
    }

    public synchronized FileContent getNewFile(HashCode fileHash) throws IOException, FileNotAvailableException {
        Preconditions.checkState(this.workingDir.isPresent());
        FileAtRelativePath file = this.getNewFileAtRelativePathOrFail(fileHash);
        Path filePath = this.getWorkingDir().resolve(file.getRelativePathString());
        return FileUtils.getFileContent(filePath);
    }

    private FileAtRelativePath getNewFileAtRelativePathOrFail(HashCode fileHash) throws IOException, FileNotAvailableException {
        Iterable<FileAtRelativePath> actualNewFiles;
        try {
            actualNewFiles = this.getNewFiles();
        }
        catch (TooManyResultFilesException e) {
            throw new FileNotAvailableException(fileHash, (Throwable)e);
        }
        for (FileAtRelativePath farp : actualNewFiles) {
            if (!farp.getFileHash().equals(fileHash)) continue;
            return farp;
        }
        throw new FileNotAvailableException(fileHash, "File does not belong to a new file.");
    }

    public int getExitCode() throws IOException {
        String asString = this.getRunInfo().get("exitcode");
        return Integer.parseInt(asString);
    }

    public TimeInterval getWallTime() throws IOException {
        String asString = this.getRunInfo().get("walltime").replace("s", "");
        Long nanoSeconds = (long)(Double.parseDouble(asString) * 1.0E9);
        return TimeInterval.nanoseconds(nanoSeconds);
    }

    public synchronized ImmutableMap<String, String> getRunInfo() throws IOException {
        File file = this.getStdoutFile().toFile();
        String content = com.google.common.io.Files.toString(file, Charsets.UTF_8);
        Map<String, String> runInfo = RUN_INFO_SPLITTER.split(content);
        return ImmutableMap.copyOf(runInfo);
    }

    public synchronized void cleanUp() {
        if (this.suppressCleanup) {
            return;
        }
        Preconditions.checkState(this.workingDir.isPresent(), "Working directory already cleaned up or not created.");
        try {
            Path tmpWorkingDir = this.workingDir.get();
            this.recursivelyDeleteDirectory(tmpWorkingDir);
            this.logger.logf(Level.FINER, "Successfully deleted working dir %s for %s.", tmpWorkingDir, this.run);
        }
        catch (IOException e) {
            this.logger.logf(Level.WARNING, "Unable to delete dir %s belonging to %s: %s", this.workingDir.get(), this.run, e.getMessage());
        }
        this.workingDir = Optional.absent();
        Preconditions.checkState(this.temporaryDir.isPresent(), "Temporary directory already cleaned up or not created.");
        try {
            Path tmpDir = this.temporaryDir.get();
            this.recursivelyDeleteDirectory(tmpDir);
            this.logger.logf(Level.FINER, "Successfully deleted temporary dir %s for %s.", tmpDir, this.run);
        }
        catch (IOException e) {
            this.logger.logf(Level.WARNING, "Unable to delete dir %s belonging to %s: %s", this.temporaryDir.get(), this.run, e.getMessage());
        }
        this.temporaryDir = Optional.absent();
    }

    private void recursivelyDeleteDirectory(Path dir) throws IOException {
        Preconditions.checkArgument(Files.isDirectory(dir, new LinkOption[0]), dir + " does not exist");
        DirDeleter deleter = new DirDeleter(this.logger);
        Files.walkFileTree(dir, deleter);
        if (Files.isDirectory(dir, new LinkOption[0])) {
            throw new IOException("Unable to delete directory for unknown reason.");
        }
    }

    private static class MemoryIsNotZeroPredicate
    implements Predicate<MemoryUnit> {
        public static final MemoryIsNotZeroPredicate INSTANCE = new MemoryIsNotZeroPredicate();

        private MemoryIsNotZeroPredicate() {
        }

        @Override
        public boolean apply(MemoryUnit input) {
            return !input.isZero();
        }
    }
}

